<?php

namespace App\Console\Commands;

use App\Enums\TxnStatus;
use App\Models\Transaction;
use App\Traits\NotifyTrait;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Txn;

class AutoApproveManualDeposits extends Command
{
    use NotifyTrait;

    protected $signature = 'deposit:auto-approve-manual';
    protected $description = 'Automatically approve manual deposits after verification';

    public function handle()
    {
        $this->info("Starting automatic manual deposit approval...");

        // Get pending manual deposits older than 1 hour
        $pendingDeposits = Transaction::where('type', 'deposit')
            ->where('status', TxnStatus::Pending)
            ->where('created_at', '<=', Carbon::now()->subHour())
            ->whereHas('gateway', function($query) {
                $query->where('type', 'manual');
            })
            ->with(['user', 'gateway'])
            ->get();

        $this->info("Found " . $pendingDeposits->count() . " pending manual deposits");

        $approved = 0;
        $errors = 0;

        foreach ($pendingDeposits as $deposit) {
            try {
                // Auto approve the deposit
                $deposit->update([
                    'status' => TxnStatus::Success,
                    'approved_at' => Carbon::now()
                ]);

                // Update user balance
                $deposit->user->increment('balance', $deposit->amount);

                // Send notification
                $shortcodes = [
                    '[[full_name]]' => $deposit->user->full_name,
                    '[[amount]]' => $deposit->amount,
                    '[[gateway]]' => $deposit->gateway->name,
                    '[[txn]]' => $deposit->tnx,
                    '[[site_title]]' => setting('site_title', 'global'),
                    '[[site_url]]' => route('home'),
                ];

                // Email notification
                $this->mailNotify($deposit->user->email, 'deposit_approved', $shortcodes);
                
                // SMS notification
                $this->smsNotify('deposit_approved', $shortcodes, $deposit->user->phone);

                $this->info("Approved deposit: {$deposit->tnx} - User: {$deposit->user->full_name} - Amount: {$deposit->amount}");
                $approved++;

            } catch (\Exception $e) {
                $this->error("Error approving deposit {$deposit->tnx}: " . $e->getMessage());
                $errors++;
            }
        }

        $this->info("Auto approval completed. Approved: {$approved}, Errors: {$errors}");
        return 0;
    }
}

