<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\RoiSchedule;
use App\Models\User;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use Txn;
use Carbon\Carbon;

class AutoRoiCollection extends Command
{
    protected $signature = 'roi:auto-collect';
    protected $description = 'Automatically collect ready ROI and distribute to users';

    public function handle()
    {
        $this->info('🚀 Starting automatic ROI collection...');
        $this->newLine();

        // Get all ready ROI schedules
        $readySchedules = RoiSchedule::where('status', 'ready')
            ->where('roi_ready_time', '<=', now())
            ->with(['user', 'invest.schema'])
            ->get();

        $this->info("Found {$readySchedules->count()} ready ROI schedules");

        if ($readySchedules->count() === 0) {
            $this->info('No ready ROI schedules found. Exiting...');
            return Command::SUCCESS;
        }

        $processedCount = 0;
        $totalAmount = 0;
        $errors = [];

        foreach ($readySchedules as $schedule) {
            try {
                $user = $schedule->user;
                $roiAmount = $schedule->roi_amount;

                if (!$user) {
                    $errors[] = "User not found for Schedule ID: {$schedule->id}";
                    continue;
                }

                // Add ROI to user's profit balance
                $user->increment('profit_balance', $roiAmount);

                // Create transaction record
                Txn::new(
                    $roiAmount,
                    0, // No charge
                    $roiAmount,
                    'system',
                    "ROI from {$schedule->invest->schema->name} Plan - Cycle {$schedule->roi_cycle}",
                    TxnType::Interest,
                    TxnStatus::Success,
                    null,
                    null,
                    $user->id
                );

                // Mark ROI as sent
                $schedule->update([
                    'status' => 'sent',
                    'roi_sent_time' => now()
                ]);

                // Create next ROI cycle if not completed
                if ($schedule->roi_cycle < $schedule->total_cycles) {
                    $nextCycle = $schedule->roi_cycle + 1;
                    
                    // Get ROI period from schema settings (in minutes)
                    $roiPeriod = $schedule->invest->schema->roi_period ?? 1440; // Default 1440 minutes (24 hours)
                    
                    RoiSchedule::create([
                        'invest_id' => $schedule->invest_id,
                        'user_id' => $schedule->user_id,
                        'schema_id' => $schedule->schema_id,
                        'invest_amount' => $schedule->invest_amount,
                        'roi_percentage' => $schedule->roi_percentage,
                        'roi_amount' => $schedule->roi_amount,
                        'roi_start_time' => now(),
                        'roi_end_time' => now()->addMinutes($roiPeriod), // Use addMinutes instead of addHours
                        'status' => 'pending',
                        'roi_cycle' => $nextCycle,
                        'total_cycles' => $schedule->total_cycles
                    ]);

                    $this->line("✅ Created next cycle ({$nextCycle}) for User: {$user->username}");
                } else {
                    // Mark investment as completed if all cycles are done
                    $schedule->invest->update(['status' => 'completed']);
                    $this->line("✅ Investment completed for User: {$user->username}");
                }

                $processedCount++;
                $totalAmount += $roiAmount;

                $this->line("✅ Processed ROI for User: {$user->username} - Amount: $" . number_format($roiAmount, 2));

            } catch (\Exception $e) {
                $errors[] = "Error processing Schedule ID: {$schedule->id} - " . $e->getMessage();
                $this->error("❌ Error processing Schedule ID: {$schedule->id} - " . $e->getMessage());
            }
        }

        $this->newLine();
        $this->info('📊 ROI Collection Summary:');
        $this->table(
            ['Metric', 'Value'],
            [
                ['Total Processed', $processedCount],
                ['Total Amount Distributed', '$' . number_format($totalAmount, 2)],
                ['Errors', count($errors)],
            ]
        );

        if (count($errors) > 0) {
            $this->newLine();
            $this->error('❌ Errors encountered:');
            foreach ($errors as $error) {
                $this->error("  - {$error}");
            }
        }

        $this->newLine();
        $this->info('✅ Automatic ROI collection completed!');
        
        return Command::SUCCESS;
    }
}
