<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\RoiSchedule;
use App\Models\Schema;

class CheckRoiCycles extends Command
{
    protected $signature = 'roi:check-cycles';
    protected $description = 'Check ROI cycles and compare with schema settings';

    public function handle()
    {
        $this->info('🔍 Checking ROI Cycles vs Schema Settings...');
        $this->newLine();

        // Get ROI schedules with their schemas
        $roiSchedules = RoiSchedule::with(['invest.schema'])
            ->select('id', 'total_cycles', 'roi_cycle', 'schema_id', 'status')
            ->limit(20)
            ->get();

        if ($roiSchedules->count() === 0) {
            $this->info('No ROI schedules found.');
            return Command::SUCCESS;
        }

        $this->table(
            ['ROI ID', 'Schema', 'Total Cycles', 'Current Cycle', 'Status', 'Schema Periods'],
            $roiSchedules->map(function ($schedule) {
                $schemaPeriods = $schedule->invest->schema->number_of_period ?? 'N/A';
                return [
                    $schedule->id,
                    $schedule->invest->schema->name ?? 'N/A',
                    $schedule->total_cycles,
                    $schedule->roi_cycle,
                    $schedule->status,
                    $schemaPeriods
                ];
            })->toArray()
        );

        $this->newLine();
        
        // Check for mismatches
        $mismatches = $roiSchedules->filter(function ($schedule) {
            $schemaPeriods = $schedule->invest->schema->number_of_period ?? 0;
            return $schedule->total_cycles != $schemaPeriods;
        });

        if ($mismatches->count() > 0) {
            $this->warn("⚠️  Found {$mismatches->count()} ROI schedules with mismatched total cycles!");
            $this->info('Run: php artisan roi:update-periods to fix this');
        } else {
            $this->info('✅ All ROI schedules match their schema periods');
        }

        $this->newLine();
        $this->info('✅ ROI Cycles check completed!');
        
        return Command::SUCCESS;
    }
}