<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use App\Models\Schema;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class FixInvestmentTimers extends Command
{
    protected $signature = 'investment:fix-timers {--dry-run : Show what would be updated without making changes}';
    protected $description = 'Fix investment timers based on proper timezone calculations';

    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        
        $this->info('=== Investment Timer Fix ===');
        
        if ($isDryRun) {
            $this->warn('DRY RUN MODE - No changes will be made');
        }
        
        // Get site timezone
        $siteTimezone = setting('site_timezone', 'global');
        $this->info("Site Timezone: {$siteTimezone}");
        
        // Get all ongoing investments
        $investments = Invest::with(['schema', 'user'])
            ->where('status', 'ongoing')
            ->get();
            
        $this->info("Found {$investments->count()} ongoing investments");
        
        $fixedCount = 0;
        $errorCount = 0;
        
        foreach ($investments as $investment) {
            try {
                $this->line("Processing Investment ID: {$investment->id}");
                
                // Get schema schedule time (in minutes)
                $scheduleMinutes = optional($investment->schema->schedule)->time ?? 1440; // Default 24 hours
                $periodHours = round($scheduleMinutes / 60, 1);
                
                $this->line("  - Schema: {$investment->schema->name}");
                $this->line("  - Period Hours: {$periodHours}");
                $this->line("  - Current next_profit_time: {$investment->next_profit_time}");
                
                // Calculate correct next profit time
                $now = Carbon::now($siteTimezone);
                $correctNextProfitTime = $now->copy()->addMinutes($scheduleMinutes);
                
                // Calculate correct last profit time (if not set, use created_at)
                $correctLastProfitTime = $investment->last_profit_time ?? $investment->created_at;
                
                $this->line("  - Correct next_profit_time: {$correctNextProfitTime}");
                $this->line("  - Correct last_profit_time: {$correctLastProfitTime}");
                
                if (!$isDryRun) {
                    // Update the investment
                    $investment->update([
                        'next_profit_time' => $correctNextProfitTime,
                        'last_profit_time' => $correctLastProfitTime,
                        'period_hours' => $periodHours,
                    ]);
                    
                    $this->info("  ✅ Updated successfully");
                } else {
                    $this->info("  📝 Would update next_profit_time to: {$correctNextProfitTime}");
                }
                
                $fixedCount++;
                
            } catch (\Exception $e) {
                $this->error("  ❌ Error processing investment {$investment->id}: " . $e->getMessage());
                $errorCount++;
            }
        }
        
        $this->info("\n=== Summary ===");
        $this->info("Fixed: {$fixedCount}");
        $this->info("Errors: {$errorCount}");
        
        if ($isDryRun) {
            $this->warn("This was a dry run. Run without --dry-run to apply changes.");
        } else {
            $this->info("Timer fixes applied successfully!");
        }
        
        // Show timezone check
        $this->info("\n=== Timezone Check ===");
        $this->info("Server Timezone: " . date_default_timezone_get());
        $this->info("Server Time: " . date('Y-m-d H:i:s'));
        $this->info("App Timezone: " . config('app.timezone'));
        $this->info("App Time: " . Carbon::now()->format('Y-m-d H:i:s'));
        $this->info("Site Timezone: {$siteTimezone}");
        $this->info("Site Time: " . Carbon::now($siteTimezone)->format('Y-m-d H:i:s'));
    }
}
