<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\RoiSchedule;
use App\Models\Invest;
use Carbon\Carbon;

class FixRoiStatus extends Command
{
    protected $signature = 'roi:fix-status';
    protected $description = 'Fix ROI status - mark expired pending as ready and create missing schedules';

    public function handle()
    {
        $this->info('🔧 Fixing ROI Status...');
        $this->newLine();

        // 1. Fix expired pending schedules to ready
        $expiredSchedules = RoiSchedule::where('status', 'pending')
            ->where('roi_end_time', '<=', now())
            ->get();

        $this->info("Found {$expiredSchedules->count()} expired pending schedules");

        $fixedCount = 0;
        foreach ($expiredSchedules as $schedule) {
            $schedule->update([
                'status' => 'ready',
                'roi_ready_time' => now()
            ]);
            $fixedCount++;
            $this->line("✅ Fixed Schedule ID: {$schedule->id} - User: {$schedule->user_id}");
        }

        $this->info("Fixed {$fixedCount} expired schedules to ready status");
        $this->newLine();

        // 2. Create missing ROI schedules for ongoing investments
        $ongoingInvestments = Invest::where('status', 'ongoing')
            ->whereDoesntHave('roiSchedules')
            ->with(['user', 'schema'])
            ->get();

        $this->info("Found {$ongoingInvestments->count()} ongoing investments without ROI schedules");

        $createdCount = 0;
        foreach ($ongoingInvestments as $investment) {
            try {
                // Calculate ROI amount
                $roiAmount = ($investment->invest_amount * $investment->schema->return_interest) / 100;
                $totalCycles = $investment->schema->number_of_period ?? 30;
                
                // Get ROI period from schema settings (in minutes)
                $roiPeriod = $investment->schema->roi_period ?? 1440; // Default 1440 minutes (24 hours)
                
                // Create ROI schedule
                RoiSchedule::create([
                    'invest_id' => $investment->id,
                    'user_id' => $investment->user_id,
                    'schema_id' => $investment->schema_id,
                    'invest_amount' => $investment->invest_amount,
                    'roi_percentage' => $investment->schema->return_interest,
                    'roi_amount' => $roiAmount,
                    'roi_start_time' => now(),
                    'roi_end_time' => now()->addMinutes($roiPeriod), // Use addMinutes instead of addHours
                    'status' => 'pending',
                    'roi_cycle' => 1,
                    'total_cycles' => $totalCycles
                ]);

                $createdCount++;
                $this->line("✅ Created ROI schedule for Investment ID: {$investment->id} - User: {$investment->user->username}");
            } catch (\Exception $e) {
                $this->error("❌ Failed to create ROI schedule for Investment ID: {$investment->id} - Error: " . $e->getMessage());
            }
        }

        $this->info("Created {$createdCount} missing ROI schedules");
        $this->newLine();

        // 3. Clean up duplicate pending schedules (keep only the first one)
        $duplicateSchedules = RoiSchedule::select('invest_id')
            ->where('status', 'pending')
            ->groupBy('invest_id')
            ->havingRaw('COUNT(*) > 1')
            ->get();

        $cleanedCount = 0;
        foreach ($duplicateSchedules as $duplicate) {
            $schedules = RoiSchedule::where('invest_id', $duplicate->invest_id)
                ->where('status', 'pending')
                ->orderBy('created_at', 'asc')
                ->get();

            // Keep the first one, delete the rest
            $keepSchedule = $schedules->first();
            $deleteSchedules = $schedules->skip(1);

            foreach ($deleteSchedules as $schedule) {
                $schedule->delete();
                $cleanedCount++;
            }

            $this->line("✅ Cleaned up duplicate schedules for Investment ID: {$duplicate->invest_id}");
        }

        $this->info("Cleaned up {$cleanedCount} duplicate schedules");
        $this->newLine();

        // 4. Show final status
        $pendingCount = RoiSchedule::where('status', 'pending')->count();
        $readyCount = RoiSchedule::where('status', 'ready')->count();
        $sentCount = RoiSchedule::where('status', 'sent')->count();

        $this->info('📊 Final ROI Status:');
        $this->table(
            ['Status', 'Count'],
            [
                ['Pending', $pendingCount],
                ['Ready', $readyCount],
                ['Sent', $sentCount],
            ]
        );

        $this->newLine();
        $this->info('✅ ROI status fix completed!');
        
        return Command::SUCCESS;
    }
}
