<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use Carbon\Carbon;

class FixSpecificInvestmentTimer extends Command
{
    protected $signature = 'investment:fix-timer {investment_id} {--hours=24 : Hours to add from now}';
    protected $description = 'Fix a specific investment timer';

    public function handle()
    {
        $investmentId = $this->argument('investment_id');
        $hours = $this->option('hours');
        
        $this->info("=== Fixing Investment Timer ===");
        $this->info("Investment ID: {$investmentId}");
        $this->info("Hours to add: {$hours}");
        
        try {
            $investment = Invest::with(['schema', 'user'])->findOrFail($investmentId);
            
            $this->info("Investment found:");
            $this->info("  - User: {$investment->user->username}");
            $this->info("  - Schema: {$investment->schema->name}");
            $this->info("  - Amount: {$investment->invest_amount}");
            $this->info("  - Current next_profit_time: {$investment->next_profit_time}");
            $this->info("  - Current last_profit_time: {$investment->last_profit_time}");
            
            // Get site timezone
            $siteTimezone = setting('site_timezone', 'global');
            $now = Carbon::now($siteTimezone);
            
            // Calculate new times
            $newNextProfitTime = $now->copy()->addHours($hours);
            $newLastProfitTime = $investment->last_profit_time ?? $investment->created_at;
            
            $this->info("  - New next_profit_time: {$newNextProfitTime}");
            $this->info("  - New last_profit_time: {$newLastProfitTime}");
            
            // Update the investment
            $investment->update([
                'next_profit_time' => $newNextProfitTime,
                'last_profit_time' => $newLastProfitTime,
                'period_hours' => $hours,
            ]);
            
            $this->info("✅ Investment timer updated successfully!");
            
            // Show time remaining
            $timeRemaining = $newNextProfitTime->diffInMinutes($now);
            $this->info("Time remaining: {$timeRemaining} minutes");
            
        } catch (\Exception $e) {
            $this->error("❌ Error: " . $e->getMessage());
        }
    }
}
