<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\RoiSchedule;
use App\Models\Schema;

class FixTestRoiPeriod extends Command
{
    protected $signature = 'roi:fix-test-period {username} {minutes=1}';
    protected $description = 'Fix ROI period for testing - set schema and all schedules to 1 minute';

    public function handle()
    {
        $username = $this->argument('username');
        $minutes = $this->argument('minutes');
        
        $this->info("🔧 Fixing ROI period to {$minutes} minute(s) for user: {$username}");
        
        // Find user
        $user = User::where('username', $username)->first();
        
        if (!$user) {
            $this->error("❌ User '{$username}' not found!");
            return Command::FAILURE;
        }
        
        $this->info("✅ User found: {$user->first_name} {$user->last_name}");
        
        // Get user's active investments
        $investments = $user->invests()->where('status', 'ongoing')->get();
        
        if ($investments->count() === 0) {
            $this->warn("⚠️  No active investments found for this user");
            return Command::SUCCESS;
        }
        
        $this->info("📊 Found {$investments->count()} active investments");
        
        foreach ($investments as $investment) {
            // Update schema ROI period to 1 minute
            $schema = $investment->schema;
            if ($schema) {
                $schema->update(['roi_period' => $minutes]);
                $this->line("✅ Updated schema '{$schema->name}' ROI period to {$minutes} minute(s)");
            }
            
            // Update ALL ROI schedules for this investment (pending, ready, sent)
            $roiSchedules = RoiSchedule::where('invest_id', $investment->id)->get();
            
            foreach ($roiSchedules as $schedule) {
                $schedule->update([
                    'roi_start_time' => now(),
                    'roi_end_time' => now()->addMinutes($minutes)
                ]);
                $this->line("✅ Updated Schedule ID: {$schedule->id} to {$minutes} minute(s)");
            }
        }
        
        $this->info("🎯 All ROI periods set to {$minutes} minute(s) for user: {$username}");
        $this->info("⏰ Next ROI will be ready in {$minutes} minute(s)");
        
        return Command::SUCCESS;
    }
}
