<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use App\Models\User;
use App\Enums\InvestStatus;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Txn;

class ProcessInvestmentProfits extends Command
{
    protected $signature = 'investment:process-profits {--dry-run : Run without making changes}';
    protected $description = 'Process investment profits and update investment status';

    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        $processedCount = 0;
        $completedCount = 0;
        
        $this->info('Starting investment profit processing...');
        
        try {
            DB::beginTransaction();
            
            // Get all ongoing investments where profit time has passed
            $investments = Invest::with(['user', 'schema'])
                ->where('status', InvestStatus::Ongoing)
                ->where('next_profit_time', '<=', Carbon::now())
                ->get();

            $this->info("Found {$investments->count()} investments ready for profit processing");

            foreach ($investments as $invest) {
                try {
                    $this->processInvestment($invest, $isDryRun);
                    $processedCount++;
                    
                    // Check if investment is completed
                    if ($invest->fresh()->status === InvestStatus::Completed) {
                        $completedCount++;
                    }
                    
                } catch (\Exception $e) {
                    $this->error("Failed to process investment ID {$invest->id}: " . $e->getMessage());
                    Log::error("Investment processing error", [
                        'investment_id' => $invest->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }
            
            if (!$isDryRun) {
                DB::commit();
                $this->info("Successfully processed {$processedCount} investments");
                $this->info("Completed {$completedCount} investments");
            } else {
                DB::rollBack();
                $this->info("DRY RUN: Would have processed {$processedCount} investments");
            }
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("Critical error in profit processing: " . $e->getMessage());
            Log::critical("Investment profit processing failed", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return 1;
        }

        return 0;
    }

    private function processInvestment(Invest $invest, bool $isDryRun = false)
    {
        $user = $invest->user;
        
        if (!$user) {
            throw new \Exception("Investment ID {$invest->id} has no valid user");
        }

        // Calculate profit amount
        $profitAmount = $this->calculateProfitAmount($invest);
        
        $this->line("Processing Investment ID {$invest->id} - User: {$user->full_name} - Profit: {$profitAmount}");

        if (!$isDryRun) {
            // Add profit to user's profit balance
            $user->increment('profit_balance', $profitAmount);
            
            // Create transaction record
            Txn::new(
                $profitAmount,
                0,
                $profitAmount,
                'system',
                "Profit from {$invest->schema->name} investment",
                TxnType::Interest,
                TxnStatus::Success,
                null,
                null,
                $user->id,
                $invest->id
            );

            // Update investment profit tracking
            $invest->increment('already_return_profit');
            $invest->update(['last_profit_time' => Carbon::now()]);

            // Check if investment should be completed
            if ($this->shouldCompleteInvestment($invest)) {
                $this->completeInvestment($invest);
            } else {
                // Set next profit time using hours-based schedule
                $periodHours = optional($invest->schema->schedule)->time
                    ?? ($invest->period_hours ?: 24);
                $invest->update([
                    'next_profit_time' => Carbon::now()->addHours($periodHours)
                ]);
            }
        }
    }

    private function calculateProfitAmount(Invest $invest): float
    {
        if ($invest->interest_type === 'percentage') {
            return ($invest->interest * $invest->invest_amount) / 100;
        }
        
        return $invest->interest;
    }

    private function shouldCompleteInvestment(Invest $invest): bool
    {
        // If return type is not period-based, continue indefinitely
        if ($invest->return_type !== 'period') {
            return false;
        }

        // Check if all periods are completed
        return $invest->already_return_profit >= $invest->number_of_period;
    }

    private function completeInvestment(Invest $invest)
    {
        $invest->update(['status' => InvestStatus::Completed]);
        
        // Return capital if applicable
        if ($invest->capital_back) {
            $invest->user->increment('balance', $invest->invest_amount);
            
            Txn::new(
                $invest->invest_amount,
                0,
                $invest->invest_amount,
                'system',
                "Capital return from {$invest->schema->name} investment",
                TxnType::CapitalReturn,
                TxnStatus::Success,
                null,
                null,
                $invest->user->id,
                $invest->id
            );
            
            $this->info("Capital returned for Investment ID {$invest->id}");
        }
        
        $this->info("Investment ID {$invest->id} completed");
    }
}
