<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class ResetDashboard extends Command
{
    protected $signature = 'dashboard:reset';
    protected $description = 'Reset all dashboard statistics to zero (preserve admin users)';

    public function handle()
    {
        $this->info('🔄 Starting Dashboard Reset...');

        try {
            // Count before deletion
            $userCount = DB::table('users')->count();
            $transactionCount = DB::table('transactions')->count();
            
            $this->info("📊 Found {$userCount} total users and {$transactionCount} transactions");

            // Confirm before proceeding
            if (!$this->confirm('⚠️  This will delete ALL users and data. Continue?')) {
                $this->error('❌ Operation cancelled by user');
                return;
            }

            // 1. Delete all users completely
            $deletedUsers = DB::table('users')->delete();
            $this->info("✅ Deleted {$deletedUsers} regular users (admin preserved)");

            // 2. Delete all transactions
            $deletedTransactions = DB::table('transactions')->delete();
            $this->info("✅ Deleted {$deletedTransactions} transactions");

            // 3. Delete investments if table exists
            if (Schema::hasTable('invests')) {
                $deletedInvestments = DB::table('invests')->delete();
                $this->info("✅ Deleted {$deletedInvestments} investments");
            }

            // 4. Delete deposits if table exists
            if (Schema::hasTable('deposits')) {
                $deletedDeposits = DB::table('deposits')->delete();
                $this->info("✅ Deleted {$deletedDeposits} deposits");
            }

            // 5. Delete withdrawals if table exists  
            if (Schema::hasTable('withdraws')) {
                $deletedWithdrawals = DB::table('withdraws')->delete();
                $this->info("✅ Deleted {$deletedWithdrawals} withdrawals");
            }

            // 6. Delete notifications
            if (Schema::hasTable('notifications')) {
                $deletedNotifications = DB::table('notifications')->delete();
                $this->info("✅ Deleted {$deletedNotifications} notifications");
            }

            // 7. Delete support tickets
            if (Schema::hasTable('support_tickets')) {
                $deletedTickets = DB::table('support_tickets')->delete();
                $this->info("✅ Deleted {$deletedTickets} support tickets");
            }

            // 8. All users deleted, no need to reset balances
            $this->info("✅ All users and data deleted");

            $this->info("\n🎉 DASHBOARD RESET COMPLETE!");
            $this->info("📊 All statistics now show ZERO");
            $this->info("👤 All users deleted - clean start");
            $this->info("🆕 Ready for new user registrations");
            $this->info("💯 Clean dashboard ready!");

        } catch (\Exception $e) {
            $this->error("❌ Error: " . $e->getMessage());
            $this->error("📝 File: " . $e->getFile() . " on line " . $e->getLine());
        }
    }
}