<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\RoiSchedule;
use Carbon\Carbon;

class SetUserRoiTimer extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'roi:set-timer {username} {minutes=1}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Set ROI timer for specific user (for testing)';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $username = $this->argument('username');
        $minutes = $this->argument('minutes');
        
        $this->info("Setting ROI timer for user: {$username} to {$minutes} minutes");
        
        // Find user
        $user = User::where('username', $username)->first();
        if (!$user) {
            $this->error("User '{$username}' not found!");
            return Command::FAILURE;
        }
        
        $this->info("User found: {$user->username} (ID: {$user->id})");
        
        // Get user's investments
        $investments = \App\Models\Invest::where('user_id', $user->id)
            ->where('status', 'ongoing')
            ->with(['schema'])
            ->get();
            
        $this->info("Found {$investments->count()} ongoing investments");
        
        foreach ($investments as $investment) {
            $this->line("Investment ID: {$investment->id}, Amount: $" . $investment->invest_amount);
            
            // Get ROI schedules for this investment
            $roiSchedules = RoiSchedule::where('invest_id', $investment->id)
                ->where('status', 'pending')
                ->get();
                
            $this->info("Found {$roiSchedules->count()} pending ROI schedules");
            
            foreach ($roiSchedules as $schedule) {
                $this->line("Schedule ID: {$schedule->id}, Status: {$schedule->status}, Cycle: {$schedule->roi_cycle}/{$schedule->total_cycles}");
                $this->line("Start Time: {$schedule->roi_start_time}");
                $this->line("End Time: {$schedule->roi_end_time}");
                
                // Update ROI end time to 1 minute from now
                $newEndTime = now()->addMinutes($minutes);
                $schedule->update([
                    'roi_start_time' => now(),
                    'roi_end_time' => $newEndTime
                ]);
                
                $this->info("Updated Schedule ID: {$schedule->id} - New End Time: {$newEndTime}");
                $this->line("---");
            }
        }
        
        $this->info("ROI timer set to {$minutes} minutes for user: {$username}");
        
        return Command::SUCCESS;
    }
}
