<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Schema;
use App\Models\RoiSchedule;

class TestSchemaChange extends Command
{
    protected $signature = 'roi:test-schema-change';
    protected $description = 'Test what happens when schema number of periods is changed';

    public function handle()
    {
        $this->info('🧪 Testing Schema Change Effect...');
        $this->newLine();

        // Get SUPER MINI USER schema
        $schema = Schema::where('name', 'SUPER MINI USER')->first();
        
        if (!$schema) {
            $this->error('SUPER MINI USER schema not found!');
            return Command::FAILURE;
        }

        $this->info("Current Schema Settings:");
        $this->table(
            ['Field', 'Value'],
            [
                ['Name', $schema->name],
                ['Number of Periods', $schema->number_of_period],
                ['ROI Period (Hours)', $schema->roi_period],
                ['Return Interest (%)', $schema->return_interest]
            ]
        );

        // Get ROI schedules for this schema
        $roiSchedules = RoiSchedule::where('schema_id', $schema->id)
            ->where('status', 'pending')
            ->limit(5)
            ->get();

        $this->newLine();
        $this->info("Current ROI Schedules (Pending):");
        if ($roiSchedules->count() > 0) {
            $this->table(
                ['ROI ID', 'Total Cycles', 'Current Cycle', 'Status'],
                $roiSchedules->map(function ($schedule) {
                    return [
                        $schedule->id,
                        $schedule->total_cycles,
                        $schedule->roi_cycle,
                        $schedule->status
                    ];
                })->toArray()
            );
        } else {
            $this->info('No pending ROI schedules found for this schema.');
        }

        $this->newLine();
        $this->info('🔧 Now changing Number of Periods from 40 to 20...');
        
        // Change the periods
        $oldPeriods = $schema->number_of_period;
        $schema->update(['number_of_period' => 20]);
        
        $this->info("✅ Changed periods from {$oldPeriods} to 20");

        $this->newLine();
        $this->info('🔄 Running roi:update-periods command...');
        
        // Run the update command
        $this->call('roi:update-periods');

        $this->newLine();
        $this->info('📊 Checking ROI schedules after update...');
        
        // Check ROI schedules again
        $updatedSchedules = RoiSchedule::where('schema_id', $schema->id)
            ->where('status', 'pending')
            ->limit(5)
            ->get();

        if ($updatedSchedules->count() > 0) {
            $this->table(
                ['ROI ID', 'Total Cycles', 'Current Cycle', 'Status'],
                $updatedSchedules->map(function ($schedule) {
                    return [
                        $schedule->id,
                        $schedule->total_cycles,
                        $schedule->roi_cycle,
                        $schedule->status
                    ];
                })->toArray()
            );
        }

        $this->newLine();
        $this->info('🔄 Reverting changes back to original...');
        $schema->update(['number_of_period' => $oldPeriods]);
        $this->info("✅ Reverted periods back to {$oldPeriods}");

        $this->newLine();
        $this->info('✅ Schema change test completed!');
        
        return Command::SUCCESS;
    }
}