<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DepositMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DepositMethodController extends Controller
{
    public function index()
    {
        $depositMethods = DepositMethod::orderBy('created_at', 'desc')->get();
        return view('admin.deposit-methods.index', compact('depositMethods'));
    }

    public function create()
    {
        return view('admin.deposit-methods.create');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'gateway_code' => 'required|string|max:255|unique:deposit_methods,gateway_code',
            'type' => 'required|in:auto,manual',
            'status' => 'required|in:0,1',
            'minimum_deposit' => 'required|numeric|min:0',
            'maximum_deposit' => 'required|numeric|gte:minimum_deposit',
            'charge' => 'required|numeric|min:0',
            'charge_type' => 'required|in:fixed,percentage',
            'currency' => 'required|string|max:10',
            'currency_symbol' => 'required|string|max:10',
            'rate' => 'required|numeric|min:0',
            'wallet_address' => 'nullable|string|max:500',
            'qr_code' => 'nullable|url',
            'payment_instructions' => 'nullable|string',
            'payment_details' => 'nullable|json',
            // Payment type fields
            'payment_type' => 'nullable|in:bank_transfer,crypto',
            'account_number' => 'nullable|string|max:255',
            'account_name' => 'nullable|string|max:255',
            'bank_name' => 'nullable|string|max:255',
            'routing_number' => 'nullable|string|max:255',
            'swift_code' => 'nullable|string|max:255',
            'crypto_type' => 'nullable|string|max:255',
            'network' => 'nullable|string|max:255',
        ], [
            'gateway_code.unique' => 'This gateway code is already in use. Please choose a different code.',
            'gateway_code.required' => 'Gateway code is required.',
            'name.required' => 'Method name is required.',
            'type.required' => 'Please select a type (Automatic or Manual).',
            'status.required' => 'Please select a status (Active or Inactive).',
        ]);

        // Post-validation: if type is manual, ensure at least one manual field is present
        $validator->after(function ($v) use ($request) {
            if ($request->input('type') === 'manual') {
                $hasAny = $request->filled('wallet_address') || $request->filled('qr_code') || $request->filled('payment_instructions') || $request->filled('payment_details');
                if (!$hasAny) {
                    $v->errors()->add('wallet_address', 'Provide at least one of wallet address, QR code, payment instructions, or raw JSON.');
                }
            }
        });

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Prepare payment details
        $paymentDetails = [];
        
        // Handle payment type specific details
        if ($request->payment_type) {
            $paymentDetails['payment_type'] = $request->payment_type;
            
            if ($request->payment_type === 'bank_transfer') {
                $paymentDetails['bank_details'] = [
                    'account_number' => $request->account_number,
                    'account_name' => $request->account_name,
                    'bank_name' => $request->bank_name,
                    'routing_number' => $request->routing_number,
                    'swift_code' => $request->swift_code,
                ];
            } elseif ($request->payment_type === 'crypto') {
                $paymentDetails['crypto_details'] = [
                    'crypto_type' => $request->crypto_type,
                    'wallet_address' => $request->wallet_address,
                    'network' => $request->network,
                    'qr_code' => $request->qr_code,
                ];
            }
        }
        
        // Add general payment details
        if ($request->wallet_address || $request->qr_code || $request->payment_instructions) {
            $paymentDetails = array_merge($paymentDetails, [
                'wallet_address' => $request->wallet_address,
                'qr_code' => $request->qr_code,
                'payment_instructions' => $request->payment_instructions
            ]);
        }

        // If payment_details JSON is provided, use it instead
        if ($request->payment_details) {
            $paymentDetails = json_decode($request->payment_details, true);
        }

        DepositMethod::create([
            'name' => $request->name,
            'gateway_code' => $request->gateway_code,
            'type' => $request->type,
            'status' => $request->status,
            'minimum_deposit' => $request->minimum_deposit,
            'maximum_deposit' => $request->maximum_deposit,
            'charge' => $request->charge,
            'charge_type' => $request->charge_type,
            'currency' => $request->currency,
            'currency_symbol' => $request->currency_symbol,
            'rate' => $request->rate,
            'payment_details' => json_encode($paymentDetails),
        ]);

        return redirect()->route('admin.deposit-methods.index')
            ->with('success', 'Deposit method created successfully.');
    }

    public function edit($id)
    {
        $depositMethod = DepositMethod::findOrFail($id);
        return view('admin.deposit-methods.edit', compact('depositMethod'));
    }

    public function update(Request $request, $id)
    {
        $depositMethod = DepositMethod::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'gateway_code' => 'required|string|max:255|unique:deposit_methods,gateway_code,' . $id,
            'type' => 'required|in:auto,manual',
            'status' => 'required|in:0,1',
            'minimum_deposit' => 'required|numeric|min:0',
            'maximum_deposit' => 'required|numeric|gte:minimum_deposit',
            'charge' => 'required|numeric|min:0',
            'charge_type' => 'required|in:fixed,percentage',
            'currency' => 'required|string|max:10',
            'currency_symbol' => 'required|string|max:10',
            'rate' => 'required|numeric|min:0',
            'wallet_address' => 'nullable|string|max:500',
            'qr_code' => 'nullable|url',
            'payment_instructions' => 'nullable|string',
            'payment_details' => 'nullable|json',
        ]);

        // Post-validation: if type is manual, ensure at least one manual field is present
        $validator->after(function ($v) use ($request) {
            if ($request->input('type') === 'manual') {
                $hasAny = $request->filled('wallet_address') || $request->filled('qr_code') || $request->filled('payment_instructions') || $request->filled('payment_details');
                if (!$hasAny) {
                    $v->errors()->add('wallet_address', 'Provide at least one of wallet address, QR code, payment instructions, or raw JSON.');
                }
            }
        });

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Prepare payment details (include bank fields and generic fields)
        $paymentDetails = [];
        // Start with JSON if provided
        if ($request->payment_details) {
            $decoded = json_decode($request->payment_details, true);
            $paymentDetails = is_array($decoded) ? $decoded : [];
        }
        // Overlay individual inputs to ensure they persist even if JSON omitted keys
        $paymentDetails['payment_type'] = $request->payment_type ?: ($paymentDetails['payment_type'] ?? ( $request->account_number || $request->bank_name ? 'bank_transfer' : null));
        if ($request->wallet_address) { $paymentDetails['wallet_address'] = $request->wallet_address; }
        if ($request->qr_code) { $paymentDetails['qr_code'] = $request->qr_code; }
        if ($request->payment_instructions !== null) { $paymentDetails['payment_instructions'] = $request->payment_instructions; }
        if ($request->account_number !== null) { $paymentDetails['account_number'] = $request->account_number; }
        if ($request->account_holder_name !== null) { $paymentDetails['account_holder_name'] = $request->account_holder_name; }
        if ($request->bank_name !== null) { $paymentDetails['bank_name'] = $request->bank_name; }
        if ($request->branch_code !== null) { $paymentDetails['branch_code'] = $request->branch_code; }

        $depositMethod->update([
            'name' => $request->name,
            'gateway_code' => $request->gateway_code,
            'type' => $request->type,
            'status' => $request->status,
            'minimum_deposit' => $request->minimum_deposit,
            'maximum_deposit' => $request->maximum_deposit,
            'charge' => $request->charge,
            'charge_type' => $request->charge_type,
            'currency' => $request->currency,
            'currency_symbol' => $request->currency_symbol,
            'rate' => $request->rate,
            'payment_details' => json_encode($paymentDetails),
        ]);

        return redirect()->route('admin.deposit-methods.index')
            ->with('success', 'Deposit method updated successfully.');
    }

    public function destroy($id)
    {
        $depositMethod = DepositMethod::findOrFail($id);
        $depositMethod->delete();

        return redirect()->route('admin.deposit-methods.index')
            ->with('success', 'Deposit method deleted successfully.');
    }
}

