<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WithdrawRequest;
use App\Models\User;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class WithdrawRequestController extends Controller
{
    public function index(Request $request)
    {
        $query = WithdrawRequest::with(['user', 'withdrawMethod', 'processedBy']);

        // Filter by status
        if ($request->has('status') && $request->status !== '') {
            $query->where('status', $request->status);
        }

        // Filter by method
        if ($request->has('method') && $request->method !== '') {
            $query->where('withdraw_method_id', $request->method);
        }

        // Search by user
        if ($request->has('search') && $request->search !== '') {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('username', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%')
                  ->orWhere('first_name', 'like', '%' . $request->search . '%')
                  ->orWhere('last_name', 'like', '%' . $request->search . '%');
            });
        }

        $requests = $query->latest()->paginate(20);
        $methods = \App\Models\WithdrawMethod::all();

        return view('admin.withdraw.requests.index', compact('requests', 'methods'));
    }

    public function show(WithdrawRequest $withdrawRequest)
    {
        $withdrawRequest->load(['user', 'withdrawMethod', 'processedBy']);
        // Pass as 'request' to the view to keep existing variable usage
        return view('admin.withdraw.requests.show', ['request' => $withdrawRequest]);
    }

    public function approve(Request $request, WithdrawRequest $withdrawRequest)
    {
        $validator = \Validator::make($request->all(), [
            'admin_note' => 'nullable|string|max:1000'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->with('error', 'Please check the form data.');
        }

        DB::beginTransaction();
        try {
            // Update withdraw request
            $withdrawRequest->update([
                'status' => 'approved',
                'admin_note' => $request->admin_note,
                'processed_by' => null,
                'processed_at' => now()
            ]);

            // Update transaction status
            $transaction = Transaction::where('details', 'like', '%"withdraw_request_id":' . $withdrawRequest->id . '%')
                ->first();
            
            if ($transaction) {
                $transaction->update(['status' => 'success']);
            }

            DB::commit();

            return redirect()->back()
                ->with('success', 'Withdraw request approved successfully!');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'Error approving withdraw request: ' . $e->getMessage());
        }
    }

    public function reject(Request $request, WithdrawRequest $withdrawRequest)
    {
        $validator = \Validator::make($request->all(), [
            'rejection_reason' => 'required|string|max:1000'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->with('error', 'Please provide a rejection reason.');
        }

        DB::beginTransaction();
        try {
            // Update withdraw request
            $withdrawRequest->update([
                'status' => 'rejected',
                'rejection_reason' => $request->rejection_reason,
                'processed_by' => null,
                'processed_at' => now()
            ]);

            // Refund amount to user
            $user = $withdrawRequest->user;
            if ($withdrawRequest->wallet_type === 'main') {
                $user->balance += $withdrawRequest->amount;
            } else {
                $user->profit_balance += $withdrawRequest->amount;
            }
            $user->save();

            // Update transaction status
            $transaction = Transaction::where('details', 'like', '%"withdraw_request_id":' . $withdrawRequest->id . '%')
                ->first();
            
            if ($transaction) {
                $transaction->update(['status' => 'failed']);
            }

            // Create refund transaction
            Transaction::create([
                'user_id' => $user->id,
                'amount' => $withdrawRequest->amount,
                'charge' => 0,
                'final_amount' => $withdrawRequest->amount,
                'tnx' => 'REF' . time() . rand(1000, 9999),
                'type' => 'refund',
                'status' => 'success',
                'method' => 'WITHDRAW-REFUND',
                'currency' => setting('site_currency', 'global'),
                'details' => json_encode([
                    'withdraw_request_id' => $withdrawRequest->id,
                    'reason' => 'Withdraw request rejected',
                    'wallet_type' => $withdrawRequest->wallet_type
                ])
            ]);

            DB::commit();

            return redirect()->back()
                ->with('success', 'Withdraw request rejected and amount refunded!');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                ->with('error', 'Error rejecting withdraw request: ' . $e->getMessage());
        }
    }

    public function complete(WithdrawRequest $withdrawRequest)
    {
        $withdrawRequest->update([
            'status' => 'completed',
            'processed_by' => null,
            'processed_at' => now()
        ]);

        return redirect()->back()
            ->with('success', 'Withdraw request marked as completed!');
    }

    public function statistics()
    {
        $stats = [
            'total_requests' => WithdrawRequest::count(),
            'pending_requests' => WithdrawRequest::where('status', 'pending')->count(),
            'approved_requests' => WithdrawRequest::where('status', 'approved')->count(),
            'rejected_requests' => WithdrawRequest::where('status', 'rejected')->count(),
            'completed_requests' => WithdrawRequest::where('status', 'completed')->count(),
            'total_amount' => WithdrawRequest::where('status', '!=', 'rejected')->sum('amount'),
            'total_fees' => WithdrawRequest::where('status', '!=', 'rejected')->sum('fee'),
        ];

        return view('admin.withdraw.statistics', compact('stats'));
    }
}

