<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Announcement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class AnnouncementController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $announcements = Announcement::with(['creator', 'updater'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('backend.announcement.index', compact('announcements'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('backend.announcement.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Check if user is authenticated
        if (!Auth::check()) {
            notify()->error('You must be logged in to create announcements.', 'Authentication Error');
            return redirect()->route('admin.login');
        }

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:announcement,offer,promotion',
            'status' => 'required|in:active,inactive',
            'show_popup' => 'nullable|boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Create data array manually instead of using $request->all()
        $data = [
            'title' => $request->input('title'),
            'message' => $request->input('message'),
            'type' => $request->input('type'),
            'status' => $request->input('status'),
            'show_popup' => $request->has('show_popup') && $request->input('show_popup') == '1',
            'start_date' => $request->input('start_date') ?: null,
            'end_date' => $request->input('end_date') ?: null,
        ];
        
        // Debug: Log the data we're creating
        \Log::info('=== ANNOUNCEMENT CREATION DEBUG START ===');
        \Log::info('Manual data array created:', $data);
        \Log::info('Auth::id(): ' . (Auth::id() ?? 'null'));
        \Log::info('Auth::check(): ' . (Auth::check() ? 'true' : 'false'));
        
        // Set created_by explicitly
        $currentUserId = Auth::id();
        if (!$currentUserId) {
            $data['created_by'] = 233;
            \Log::warning('Using fallback user ID for announcement creation: ' . $data['created_by']);
        } else {
            $data['created_by'] = $currentUserId;
        }
        
        // Add timestamps
        $data['created_at'] = now();
        $data['updated_at'] = now();
        
        // Debug: Log the final data
        \Log::info('Final data for announcement creation:', $data);
        \Log::info('=== ANNOUNCEMENT CREATION DEBUG END ===');

        try {
            // Use DB::table() instead of Eloquent to bypass any model interference
            $id = DB::table('announcements')->insertGetId($data);
            
            if ($id) {
                notify()->success('Announcement created successfully!', 'Success');
                return redirect()->route('admin.announcement.index');
            } else {
                throw new \Exception('Failed to insert announcement');
            }
        } catch (\Exception $e) {
            \Log::error('Error creating announcement: ' . $e->getMessage());
            \Log::error('Data that caused error:', $data);
            notify()->error('Error creating announcement: ' . $e->getMessage(), 'Error');
            return redirect()->back()->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Announcement $announcement)
    {
        return view('backend.announcement.show', compact('announcement'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Announcement $announcement)
    {
        return view('backend.announcement.edit', compact('announcement'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Announcement $announcement)
    {
        // Check if user is authenticated
        if (!Auth::check()) {
            notify()->error('You must be logged in to update announcements.', 'Authentication Error');
            return redirect()->route('admin.login');
        }

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|in:announcement,offer,promotion',
            'status' => 'required|in:active,inactive',
            'show_popup' => 'nullable|boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Create data array manually
        $data = [
            'title' => $request->input('title'),
            'message' => $request->input('message'),
            'type' => $request->input('type'),
            'status' => $request->input('status'),
            'show_popup' => $request->has('show_popup') && $request->input('show_popup') == '1',
            'start_date' => $request->input('start_date') ?: null,
            'end_date' => $request->input('end_date') ?: null,
        ];
        
        // Set updated_by explicitly
        $currentUserId = Auth::id();
        if (!$currentUserId) {
            $data['updated_by'] = 233;
            \Log::warning('Using fallback user ID for announcement update: ' . $data['updated_by']);
        } else {
            $data['updated_by'] = $currentUserId;
        }

        try {
            \Log::info('Updating announcement with data:', $data);
            $announcement->update($data);
            notify()->success('Announcement updated successfully!', 'Success');
            return redirect()->route('admin.announcement.index');
        } catch (\Exception $e) {
            \Log::error('Error updating announcement: ' . $e->getMessage());
            \Log::error('Data that caused error:', $data);
            notify()->error('Error updating announcement: ' . $e->getMessage(), 'Error');
            return redirect()->back()->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Announcement $announcement)
    {
        try {
            $announcement->delete();
            notify()->success('Announcement deleted successfully!', 'Success');
        } catch (\Exception $e) {
            \Log::error('Error deleting announcement: ' . $e->getMessage());
            notify()->error('Error deleting announcement. Please try again.', 'Error');
        }
        
        return redirect()->route('admin.announcement.index');
    }

    /**
     * Toggle announcement status
     */
    public function toggleStatus(Announcement $announcement)
    {
        try {
            // Get the current user ID or use the correct fallback user ID (233)
            $currentUserId = Auth::id();
            if (!$currentUserId) {
                $currentUserId = 233;
            }
            
            $announcement->update([
                'status' => $announcement->status === 'active' ? 'inactive' : 'active',
                'updated_by' => $currentUserId
            ]);

            $status = $announcement->status === 'active' ? 'activated' : 'deactivated';
            notify()->success("Announcement {$status} successfully!", 'Success');
        } catch (\Exception $e) {
            \Log::error('Error toggling announcement status: ' . $e->getMessage());
            notify()->error('Error updating announcement status. Please try again.', 'Error');
        }
        
        return redirect()->back();
    }

    /**
     * Toggle popup status
     */
    public function togglePopup(Announcement $announcement)
    {
        try {
            // Get the current user ID or use the correct fallback user ID (233)
            $currentUserId = Auth::id();
            if (!$currentUserId) {
                $currentUserId = 233;
            }
            
            $announcement->update([
                'show_popup' => !$announcement->show_popup,
                'updated_by' => $currentUserId
            ]);

            $popup = $announcement->show_popup ? 'enabled' : 'disabled';
            notify()->success("Popup {$popup} for this announcement!", 'Success');
        } catch (\Exception $e) {
            \Log::error('Error toggling announcement popup: ' . $e->getMessage());
            notify()->error('Error updating announcement popup. Please try again.', 'Error');
        }
        
        return redirect()->back();
    }
}
