<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Invest;
use App\Models\Schema;
use App\Models\User;
use App\Models\RoiSchedule;
use Illuminate\Http\Request;
use Carbon\Carbon;

class RoiController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:admin');
    }
    
    public function index()
    {
        // Get all active investments grouped by schema
        $investments = Invest::with(['user', 'schema'])
            ->where('status', 'ongoing')
            ->get()
            ->groupBy('schema_id');
            
        // Get all schemas for dropdown
        $schemas = Schema::where('status', 1)->get();
            
        // Get ROI statistics
        $roiStats = [
            'total_schedules' => RoiSchedule::count(),
            'pending_schedules' => RoiSchedule::where('status', 'pending')->count(),
            'ready_schedules' => RoiSchedule::where('status', 'ready')->count(),
            'sent_schedules' => RoiSchedule::where('status', 'sent')->count(),
            'completed_schedules' => RoiSchedule::where('status', 'completed')->count(),
            'pending_amount' => RoiSchedule::where('status', 'pending')->sum('roi_amount'),
            'ready_amount' => RoiSchedule::where('status', 'ready')->sum('roi_amount'),
            'sent_amount' => RoiSchedule::where('status', 'sent')->sum('roi_amount'),
        ];
            
        return view('backend.roi.index', compact('investments', 'schemas', 'roiStats'));
    }
    
    public function getPlanUsers(Request $request)
    {
        try {
            $schemaId = $request->schema_id;
            
            if (!$schemaId) {
                return response()->json([
                    'error' => 'Schema ID is required',
                    'users' => []
                ], 400);
            }
            
            $investments = Invest::with(['user', 'schema', 'roiSchedules'])
                ->where('schema_id', $schemaId)
                ->where('status', 'ongoing')
                ->get();
                
            $users = $investments->map(function($investment) {
                try {
                    $activeSchedule = $investment->roiSchedules->where('status', 'pending')->first();
                    $readySchedule = $investment->roiSchedules->where('status', 'ready')->first();
                    
                    return [
                        'id' => $investment->user ? $investment->user->id : 0,
                        'name' => $investment->user ? $investment->user->full_name : 'Unknown',
                        'username' => $investment->user ? $investment->user->username : 'unknown',
                        'invest_amount' => $investment->invest_amount,
                        'roi_amount' => $investment->schema ? ($investment->invest_amount * $investment->schema->return_interest) / 100 : 0,
                        'roi_percentage' => $investment->schema ? $investment->schema->return_interest : 0,
                        'roi_status' => $readySchedule ? 'ready' : ($activeSchedule ? 'pending' : 'none'),
                        'time_remaining' => $activeSchedule ? $activeSchedule->getTimeRemaining() : 0,
                        'progress_percentage' => $activeSchedule ? $activeSchedule->getProgressPercentage() : 0,
                        'roi_cycle' => $activeSchedule ? $activeSchedule->roi_cycle : ($readySchedule ? $readySchedule->roi_cycle : 0),
                        'total_cycles' => $activeSchedule ? $activeSchedule->total_cycles : ($readySchedule ? $readySchedule->total_cycles : 0)
                    ];
                } catch (\Exception $e) {
                    \Log::error('Error processing investment: ' . $e->getMessage());
                    return null;
                }
            })->filter(); // Remove null values
            
            return response()->json([
                'success' => true,
                'users' => $users,
                'roi_percentage' => $investments->first() && $investments->first()->schema ? $investments->first()->schema->return_interest : 0
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error in getPlanUsers: ' . $e->getMessage());
            
            return response()->json([
                'error' => 'Failed to load users: ' . $e->getMessage(),
                'users' => []
            ], 500);
        }
    }
}