<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\WithdrawMethod;
use App\Models\WithdrawRequest;
use App\Models\Transaction;
use App\Enums\TxnType;
use App\Enums\TxnStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;

class NewWithdrawController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $currencySymbol = setting('site_currency_symbol', 'global', '$');
        
        // Get withdraw methods grouped by type
        $query = WithdrawMethod::query();
        
        // Check if is_active column exists
        if (Schema::hasColumn('withdraw_methods', 'is_active')) {
            $query->where('is_active', true);
        }
        
        // Check if sort_order column exists
        if (Schema::hasColumn('withdraw_methods', 'sort_order')) {
            $query->orderBy('sort_order');
        }
        
        $query->orderBy('name');
        
        $methods = $query->get()->groupBy('type');

        return view('frontend.shahdeveloper.user.withdraw.new', compact('user', 'currencySymbol', 'methods'));
    }

    public function store(Request $request)
    {
        try {
            Log::info('New withdraw request started', [
                'all_data' => $request->all(),
                'method' => $request->method(),
                'url' => $request->url(),
                'headers' => $request->headers->all()
            ]);

            $user = Auth::user();

            // Basic validation (manual to ensure JSON response instead of redirect)
            $validator = \Validator::make($request->all(), [
                'withdraw_method_id' => 'required|exists:withdraw_methods,id',
                'amount' => 'required|numeric|min:0.01',
                'wallet_type' => 'required|in:main,profit',
                'method_details' => 'nullable|array'
            ], [
                'withdraw_method_id.required' => 'Please select a withdrawal method.',
                'withdraw_method_id.exists' => 'Selected withdrawal method is invalid.',
                'amount.required' => 'Please enter an amount to withdraw.',
                'amount.numeric' => 'Withdrawal amount must be a valid number.',
                'amount.min' => 'Withdrawal amount must be at least 0.01.',
                'wallet_type.required' => 'Please choose a wallet to withdraw from.',
                'wallet_type.in' => 'Selected wallet is invalid.',
                'method_details.array' => 'Payment details must be sent as a valid set of fields.'
            ]);

            if ($validator->fails()) {
                Log::warning('Withdraw validation failed', $validator->errors()->toArray());
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'errors' => $validator->errors()->toArray()
                ], 422);
            }

            // Get withdraw method
            $withdrawMethod = WithdrawMethod::findOrFail($request->withdraw_method_id);

            // Check if method is active (if column exists)
            if (Schema::hasColumn('withdraw_methods', 'is_active') && !$withdrawMethod->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Selected withdrawal method is not available.'
                ]);
            }

            $amount = $request->amount;

            // Check minimum and maximum amounts
            if ($amount < $withdrawMethod->min_amount) {
                return response()->json([
                    'success' => false,
                    'message' => "Minimum withdrawal amount is {$withdrawMethod->min_amount}."
                ]);
            }

            if ($amount > $withdrawMethod->max_amount) {
                return response()->json([
                    'success' => false,
                    'message' => "Maximum withdrawal amount is {$withdrawMethod->max_amount}."
                ]);
            }

            // Check user balance
            if ($request->wallet_type === 'main' && $user->balance < $amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance in Main Wallet.'
                ]);
            }

            if ($request->wallet_type === 'profit' && $user->profit_balance < $amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance in Profit Wallet.'
                ]);
            }

            // Calculate fee
            $fee = 0;
            if (Schema::hasColumn('withdraw_methods', 'fee_percentage') && Schema::hasColumn('withdraw_methods', 'fee_fixed')) {
                $fee = $withdrawMethod->calculateFee($amount);
            }
            $finalAmount = $amount - $fee;

            // Use database transaction to ensure atomicity
            DB::beginTransaction();
            
            try {
                // Create withdraw request
                $withdrawRequest = WithdrawRequest::create([
                    'user_id' => $user->id,
                    'withdraw_method_id' => $withdrawMethod->id,
                    'amount' => $amount,
                    'fee' => $fee,
                    'final_amount' => $finalAmount,
                    'wallet_type' => $request->wallet_type,
                    'method_details' => $request->input('method_details', []),
                    'status' => 'pending'
                ]);

                // Create transaction record
                $transaction = Transaction::create([
                    'user_id' => $user->id,
                    'amount' => $amount,
                    'charge' => $fee,
                    'final_amount' => $finalAmount,
                    'tnx' => 'WDR' . time() . rand(1000, 9999),
                    'type' => TxnType::Withdraw,
                    'status' => TxnStatus::Pending,
                    'method' => 'WITHDRAW-' . strtoupper($withdrawMethod->type),
                    'currency' => setting('site_currency', 'global'),
                    'description' => 'Withdrawal Request - ' . $withdrawMethod->name,
                    'details' => json_encode([
                        'withdraw_request_id' => $withdrawRequest->id,
                        'method_name' => $withdrawMethod->name,
                        'wallet_type' => $request->wallet_type
                    ])
                ]);

                // Deduct balance from user
                if ($request->wallet_type === 'main') {
                    Log::info('Deducting from main balance', [
                        'user_id' => $user->id,
                        'balance_before' => $user->balance,
                        'amount' => $amount,
                        'balance_after' => $user->balance - $amount
                    ]);
                    
                    // Try to update users table first
                    if (Schema::hasColumn('users', 'balance')) {
                        $newBalance = $user->balance - $amount;
                        $updated = DB::table('users')
                            ->where('id', $user->id)
                            ->update(['balance' => $newBalance]);
                        
                        Log::info('Balance updated in users table', [
                            'user_id' => $user->id,
                            'old_balance' => $user->balance,
                            'amount' => $amount,
                            'new_balance' => $newBalance,
                            'rows_affected' => $updated
                        ]);
                        
                        // Verify the update
                        $verifyBalance = DB::table('users')->where('id', $user->id)->value('balance');
                        Log::info('Balance verification', [
                            'user_id' => $user->id,
                            'verified_balance' => $verifyBalance
                        ]);
                    } else {
                        Log::info('Balance column does not exist in users table, trying user_wallets');
                        // Update user_wallets table
                        $usdWallet = $user->wallets()->where('wallet_type', 'USD')->first();
                        if ($usdWallet) {
                            $usdWallet->deductBalance($amount);
                            Log::info('Balance deducted from user_wallets', [
                                'user_id' => $user->id,
                                'wallet_id' => $usdWallet->id,
                                'new_balance' => $usdWallet->balance
                            ]);
                        } else {
                            Log::error('USD wallet not found for user', ['user_id' => $user->id]);
                        }
                    }
                    
                    // Refresh user to get updated balance
                    $user->refresh();
                    
                    Log::info('Balance after refresh', [
                        'user_id' => $user->id,
                        'actual_balance' => $user->balance
                    ]);
                    
                } else {
                    Log::info('Deducting from profit balance', [
                        'user_id' => $user->id,
                        'profit_balance_before' => $user->profit_balance,
                        'amount' => $amount,
                        'profit_balance_after' => $user->profit_balance - $amount
                    ]);
                    
                    // Try to update users table first
                    if (Schema::hasColumn('users', 'profit_balance')) {
                        $newProfitBalance = $user->profit_balance - $amount;
                        $updated = DB::table('users')
                            ->where('id', $user->id)
                            ->update(['profit_balance' => $newProfitBalance]);
                        
                        Log::info('Profit balance updated in users table', [
                            'user_id' => $user->id,
                            'new_profit_balance' => $newProfitBalance,
                            'rows_affected' => $updated
                        ]);
                    } else {
                        // Update user_wallets table
                        $profitWallet = $user->wallets()->where('wallet_type', 'PROFIT')->first();
                        if ($profitWallet) {
                            $profitWallet->deductBalance($amount);
                            Log::info('Profit balance deducted from user_wallets', [
                                'user_id' => $user->id,
                                'wallet_id' => $profitWallet->id,
                                'new_balance' => $profitWallet->balance
                            ]);
                        } else {
                            Log::error('PROFIT wallet not found for user', ['user_id' => $user->id]);
                        }
                    }
                    
                    // Refresh user to get updated balance
                    $user->refresh();
                    
                    Log::info('Profit balance after refresh', [
                        'user_id' => $user->id,
                        'actual_profit_balance' => $user->profit_balance
                    ]);
                }
                
                // Commit the transaction
                DB::commit();
                
                // Log balance after save
                $user->refresh();
                Log::info('Balance after withdrawal save', [
                    'user_id' => $user->id,
                    'main_balance' => $user->balance,
                    'profit_balance' => $user->profit_balance
                ]);
                
            } catch (\Exception $e) {
                // Rollback the transaction on error
                DB::rollback();
                
                Log::error('Withdraw transaction failed', [
                    'user_id' => $user->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Withdrawal failed. Please try again.'
                ], 500);
            }

            Log::info('Withdraw request created successfully', [
                'withdraw_request_id' => $withdrawRequest->id,
                'transaction_id' => $transaction->id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Withdrawal request submitted successfully! Admin will process it soon.',
                'withdraw_request_id' => $withdrawRequest->id
            ]);

        } catch (\Throwable $e) {
            Log::error('Withdraw request error: ' . $e->getMessage(), [
                'line' => $e->getLine(),
                'file' => $e->getFile(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function history()
    {
        $user = Auth::user();
        $withdrawRequests = WithdrawRequest::where('user_id', $user->id)
            ->with('withdrawMethod')
            ->latest()
            ->paginate(20);

        return view('frontend.shahdeveloper.user.withdraw.history', compact('withdrawRequests'));
    }
}
