<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Ranking;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RankingController extends Controller
{
    /**
     * Display the ranks and rewards page
     */
    public function index()
    {
        // Get all active rankings ordered by ranking level
        $rankings = Ranking::where('status', 1)
            ->orderBy('ranking', 'asc')
            ->get();

        // Get user's current ranking if logged in
        $userRanking = null;
        $userProgress = null;
        
        if (Auth::check()) {
            $user = Auth::user();
            $userRanking = $user->ranking;
            
            // Calculate user's progress towards next rank
            if ($userRanking && $userRanking->ranking > 0) {
                $nextRank = Ranking::where('status', 1)
                    ->where('ranking', '>', $userRanking->ranking)
                    ->orderBy('ranking', 'asc')
                    ->first();
                
                if ($nextRank) {
                    $userProgress = [
                        'current_rank' => $userRanking,
                        'next_rank' => $nextRank,
                        'progress_percentage' => $this->calculateProgress($user, $nextRank)
                    ];
                }
            }
        }

        $currencySymbol = setting('currency_symbol', 'global', '$');
        
        return view('frontend.shahdeveloper.user.ranking.index', compact(
            'rankings', 
            'userRanking', 
            'userProgress', 
            'currencySymbol'
        ));
    }

    /**
     * Calculate user's progress towards next rank
     */
    private function calculateProgress($user, $nextRank)
    {
        $progress = 0;
        
        // Calculate earnings progress
        if ($nextRank->minimum_earnings > 0) {
            $earningsProgress = min(($user->total_earnings ?? 0) / $nextRank->minimum_earnings * 100, 100);
            $progress = max($progress, $earningsProgress);
        }
        
        // Calculate deposit progress
        if ($nextRank->minimum_deposit > 0) {
            $depositProgress = min(($user->total_deposit ?? 0) / $nextRank->minimum_deposit * 100, 100);
            $progress = max($progress, $depositProgress);
        }
        
        // Calculate investment progress
        if ($nextRank->minimum_invest > 0) {
            $investProgress = min(($user->total_invest ?? 0) / $nextRank->minimum_invest * 100, 100);
            $progress = max($progress, $investProgress);
        }
        
        // Calculate referral progress
        if ($nextRank->minimum_referral > 0) {
            $referralProgress = min(($user->total_referral ?? 0) / $nextRank->minimum_referral * 100, 100);
            $progress = max($progress, $referralProgress);
        }
        
        return round($progress, 2);
    }
}
