<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\ReferralLink;
use App\Models\ReferralProgram;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ReferralTreeController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        
        // Get user's direct referrals
        $directReferrals = User::where('ref_id', $user->id)
            ->with(['referrals' => function($query) {
                $query->withCount('referrals');
            }])
            ->get();
        
        // Get all referrals in tree structure
        $referralTree = $this->buildReferralTree($user);
        
        // Calculate statistics
        $totalReferrals = $this->getTotalReferrals($user);
        $activeReferrals = $this->getActiveReferrals($user);
        $totalInvestment = $this->getTotalInvestment($user);
        $totalEarnings = $this->getTotalEarnings($user);
        
        // Get referral levels
        $referralLevels = $this->getReferralLevels($user);
        
        // Get referral link data
        $getReferral = ReferralLink::getReferral($user, ReferralProgram::first());
        $getReferral->direct_link = route('register') . '?invite=' . $getReferral->code;
        
        return view('frontend.shahdeveloper.referral-tree.index', compact(
            'user',
            'directReferrals',
            'referralTree',
            'totalReferrals',
            'activeReferrals',
            'totalInvestment',
            'totalEarnings',
            'referralLevels',
            'getReferral'
        ));
    }
    
    /**
     * Build referral tree structure
     */
    private function buildReferralTree($user, $level = 0, $maxLevel = 5)
    {
        if ($level >= $maxLevel) {
            return [];
        }
        
        $referrals = User::where('ref_id', $user->id)
            ->with(['referrals' => function($query) {
                $query->withCount('referrals');
            }])
            ->get()
            ->map(function($referral) use ($level, $maxLevel) {
                $referral->level = $level + 1;
                $referral->children = $this->buildReferralTree($referral, $level + 1, $maxLevel);
                $referral->total_investment = $referral->totalInvestment();
                $referral->total_deposit = $referral->totalDeposit();
                $referral->total_profit = $referral->totalProfit();
                $referral->referral_count = $referral->referrals()->count();
                return $referral;
            });
        
        return $referrals;
    }
    
    /**
     * Get total referrals count
     */
    private function getTotalReferrals($user)
    {
        return User::where('ref_id', $user->id)->count();
    }
    
    /**
     * Get active referrals count
     */
    private function getActiveReferrals($user)
    {
        return User::where('ref_id', $user->id)
            ->where('status', 1)
            ->count();
    }
    
    /**
     * Get total investment from referrals
     */
    private function getTotalInvestment($user)
    {
        $referrals = User::where('ref_id', $user->id)->get();
        $totalInvestment = 0;
        
        foreach($referrals as $referral) {
            $totalInvestment += $referral->totalInvestment();
        }
        
        return $totalInvestment;
    }
    
    /**
     * Get total earnings from referrals
     */
    private function getTotalEarnings($user)
    {
        $referrals = User::where('ref_id', $user->id)->get();
        $totalEarnings = 0;
        
        foreach($referrals as $referral) {
            $totalEarnings += $referral->totalProfit();
        }
        
        return $totalEarnings;
    }
    
    /**
     * Get referral levels data
     */
    private function getReferralLevels($user)
    {
        $levels = [];
        
        for ($i = 1; $i <= 10; $i++) {
            $levelReferrals = $this->getReferralsByLevel($user, $i);
            $investment = 0;
            $deposit = 0;
            $profit = 0;
            
            foreach($levelReferrals as $referral) {
                $investment += $referral->totalInvestment();
                $deposit += $referral->totalDeposit();
                $profit += $referral->totalProfit();
            }
            
            $levels[$i] = [
                'count' => $levelReferrals->count(),
                'investment' => $investment,
                'deposit' => $deposit,
                'profit' => $profit,
            ];
        }
        
        return $levels;
    }
    
    /**
     * Get referrals by specific level
     */
    private function getReferralsByLevel($user, $level)
    {
        if ($level == 1) {
            return User::where('ref_id', $user->id)->get();
        }
        
        $referralIds = [$user->id];
        
        for ($i = 1; $i < $level; $i++) {
            $referralIds = User::whereIn('ref_id', $referralIds)->pluck('id')->toArray();
            if (empty($referralIds)) {
                break;
            }
        }
        
        return User::whereIn('ref_id', $referralIds)->get();
    }
    
    /**
     * Get referral details for AJAX
     */
    public function getReferralDetails(Request $request)
    {
        $referralId = $request->input('referral_id');
        $referral = User::find($referralId);
        
        if (!$referral) {
            return response()->json(['error' => 'Referral not found'], 404);
        }
        
        $referral->total_investment = $referral->totalInvestment();
        $referral->total_deposit = $referral->totalDeposit();
        $referral->total_profit = $referral->totalProfit();
        $referral->referral_count = $referral->referrals()->count();
        $referral->join_date = $referral->created_at->format('M d, Y');
        $referral->status_text = $referral->status ? 'Active' : 'Inactive';
        
        return response()->json($referral);
    }
}
