<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Traits\ImageUpload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use PragmaRX\Google2FALaravel\Support\Authenticator;

class SettingController extends Controller
{
    use ImageUpload;

    public function settings()
    {
        // Check if shahdeveloper theme is active
        if (setting('active_theme', 'theme') === 'shahdeveloper') {
            return view('frontend.shahdeveloper.user.setting.index');
        }
        
        return view('frontend::user.setting.index');
    }

    public function profileUpdate(Request $request)
    {
        $input = $request->all();
        $user = \Auth::user();
        
        // Debug: Log request data
        \Log::info('Profile Update Request:', [
            'has_avatar_file' => $request->hasFile('avatar'),
            'avatar_file' => $request->file('avatar'),
            'user_id' => $user->id,
            'current_avatar' => $user->avatar
        ]);
        
        $validator = Validator::make($request->all(), [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'username' => 'required|string|max:255|unique:users,username,' . $user->id,
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string|max:20',
            'country' => 'required|string|max:255',
            'city' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:500',
            'zip_code' => 'nullable|string|max:20',
            'gender' => 'nullable|string|in:male,female,other',
            'date_of_birth' => 'nullable|date|before:today',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        // Handle avatar upload
        $avatarPath = $user->avatar;
        if ($request->hasFile('avatar')) {
            try {
                $avatarPath = self::imageUploadTrait($input['avatar'], $user->avatar);
                \Log::info('Avatar uploaded successfully:', ['path' => $avatarPath]);
            } catch (\Exception $e) {
                \Log::error('Avatar upload failed:', ['error' => $e->getMessage()]);
                notify()->error('Avatar upload failed: ' . $e->getMessage(), 'Error');
                return redirect()->back();
            }
        }

        $data = [
            'avatar' => $avatarPath,
            'first_name' => $input['first_name'],
            'last_name' => $input['last_name'],
            'username' => $input['username'],
            'email' => $input['email'],
            'phone' => $input['phone'],
            'country' => $input['country'],
            'city' => $input['city'] ?? null,
            'zip_code' => $input['zip_code'] ?? null,
            'address' => $input['address'] ?? null,
            'gender' => $input['gender'] ?? null,
            'date_of_birth' => $input['date_of_birth'] == '' ? null : $input['date_of_birth'],
        ];

        $user->update($data);

        notify()->success('Your Profile Updated successfully');
        return redirect()->route('user.setting.show');
    }

    public function twoFa()
    {
        $user = \Auth::user();
        $google2fa = app('pragmarx.google2fa');
        $secret = $google2fa->generateSecretKey();

        $user->update([
            'google2fa_secret' => $secret,
        ]);

        notify()->success(__('QR Code And Secret Key Generate successfully'));
        return redirect()->back();
    }

    public function actionTwoFa(Request $request)
    {
        $user = \Auth::user();

        if ($request->status == 'disable') {
            if (Hash::check(request('one_time_password'), $user->password)) {
                $user->update([
                    'two_fa' => 0,
                ]);

                notify()->success(__('2Fa Authentication Disable successfully'));
                return redirect()->back();
            }

            notify()->warning(__('Wrong Your Password'));
            return redirect()->back();
        } elseif ($request->status == 'enable') {
            session([
                config('google2fa.session_var') => [
                    'auth_passed' => false,
                ],
            ]);

            $authenticator = app(Authenticator::class)->boot($request);
            if ($authenticator->isAuthenticated()) {
                $user->update([
                    'two_fa' => 1,
                ]);

                notify()->success(__('2Fa Authentication Enable successfully'));
                return redirect()->back();
            }

            notify()->warning(__('2Fa Authentication Wrong One Time Key'));
            return redirect()->back();
        }
    }

    public function showFundPasswordForm()
    {
        return view('user.set_fund_password');
    }

    public function setFundPassword(Request $request)
    {
        $request->validate([
            'fund_password' => 'required|string|min:4|max:10',
            'fund_password_confirmation' => 'required|same:fund_password',
        ]);

        $user = auth()->user();
        $user->fund_password = Hash::make($request->fund_password);
        $user->save();

        return back()->with('success', 'T-PIN set successfully.');
    }
}