<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Transaction;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ShahTransferController extends Controller
{
    public function index()
    {
        return view('frontend.shahdeveloper.user.transfer.index');
    }
    
    public function transfer(Request $request)
    {
        Log::info('Transfer request started', $request->all());
        try {
            // Use manual validator so we can return JSON instead of redirect (avoids page reload)
            $validator = Validator::make($request->all(), [
                'recipient' => 'required|string',
                'amount' => 'required|numeric|min:0.01',
                'wallet_type' => 'required|in:main,profit',
                'password' => 'required|string',
                'note' => 'nullable|string|max:255'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ]);
            }

            $user = Auth::user();
            
            // Verify Account Password
            if (!Hash::check($request->password, $user->password)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid account password'
                ]);
            }

            // Find recipient
            $recipient = User::where('username', $request->recipient)
                ->orWhere('email', $request->recipient)
                ->first();

            if (!$recipient) {
                return response()->json([
                    'success' => false,
                    'message' => 'Recipient not found'
                ]);
            }

            if ($recipient->id === $user->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot transfer to yourself'
                ]);
            }

            $amount = $request->amount;
            $walletType = $request->wallet_type;
            $fee = 0; // No fee for transfers

            // Check balance based on selected wallet
            if ($walletType === 'main' && $user->balance < $amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance in Main Wallet'
                ]);
            } elseif ($walletType === 'profit' && $user->profit_balance < $amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance in Profit Wallet'
                ]);
            }

            // Create transaction for sender
            $senderTransaction = new Transaction();
            $senderTransaction->user_id = $user->id;
            $senderTransaction->amount = $amount;
            $senderTransaction->charge = $fee;
            $senderTransaction->final_amount = $amount;
            $senderTransaction->tnx = 'TXN' . time() . rand(1000, 9999);
            $senderTransaction->type = 'send_money';
            $senderTransaction->status = 'success';
            $senderTransaction->method = 'TRANSFER-' . strtoupper($walletType);
            $senderTransaction->currency = setting('site_currency', 'global');
            $senderTransaction->details = json_encode([
                'recipient' => $recipient->username,
                'wallet_type' => $walletType,
                'note' => $request->note
            ]);
            $senderTransaction->save();

            // Create transaction for recipient
            $recipientTransaction = new Transaction();
            $recipientTransaction->user_id = $recipient->id;
            $recipientTransaction->amount = $amount;
            $recipientTransaction->charge = 0;
            $recipientTransaction->final_amount = $amount;
            $recipientTransaction->tnx = 'TXN' . time() . rand(1000, 9999);
            $recipientTransaction->type = 'receive_money';
            $recipientTransaction->status = 'success';
            $recipientTransaction->method = 'TRANSFER-RECEIVE';
            $recipientTransaction->currency = setting('site_currency', 'global');
            $recipientTransaction->details = json_encode([
                'sender' => $user->username,
                'from_wallet' => $walletType,
                'note' => $request->note
            ]);
            $recipientTransaction->save();

            // Update sender balance based on selected wallet
            if ($walletType === 'main') {
                $user->balance -= $amount;
            } else {
                $user->profit_balance -= $amount;
            }
            $user->save();

            // Add to recipient's main wallet
            $recipient->balance += $amount;
            $recipient->save();

            return response()->json([
                'success' => true,
                'message' => 'Transfer completed successfully! Amount sent from ' . ucfirst($walletType) . ' Wallet.'
            ]);

        } catch (\Exception $e) {
            Log::error('Transfer Error: ' . $e->getMessage(), [
                'line' => $e->getLine(),
                'file' => $e->getFile(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Transfer Error: ' . $e->getMessage()
            ]);
        }
    }
    
    public function history()
    {
        $transactions = Transaction::where('user_id', Auth::id())
            ->whereIn('type', ['transfer', 'receive_money'])
            ->latest()
            ->paginate(20);
        
        return view('frontend.shahdeveloper.user.transfer.history', compact('transactions'));
    }
    
    public function walletExchange(Request $request)
    {
        try {
            $request->validate([
                'from_wallet' => 'required|in:1,2',
                'to_wallet' => 'required|in:1,2',
                'amount' => 'required|numeric|min:0.01'
            ]);

            $user = Auth::user();
            $amount = $request->amount;
            $fromWallet = $request->from_wallet;
            $toWallet = $request->to_wallet;

            // Check if same wallet
            if ($fromWallet === $toWallet) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot exchange to the same wallet'
                ]);
            }

            // Check balance
            if ($fromWallet == 1 && $user->balance < $amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance in Main Wallet'
                ]);
            } elseif ($fromWallet == 2 && $user->profit_balance < $amount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient balance in Profit Wallet'
                ]);
            }

            // Perform exchange
            if ($fromWallet == 1) {
                $user->balance -= $amount;
                $user->profit_balance += $amount;
            } else {
                $user->profit_balance -= $amount;
                $user->balance += $amount;
            }
            $user->save();

            // Create transaction record
            $transaction = new Transaction();
            $transaction->user_id = $user->id;
            $transaction->amount = $amount;
            $transaction->charge = 0;
            $transaction->final_amount = $amount;
            $transaction->tnx = 'TXN' . time() . rand(1000, 9999);
            $transaction->type = 'exchange';
            $transaction->status = 'success';
            $transaction->method = 'WALLET-EXCHANGE';
            $transaction->currency = setting('site_currency', 'global');
            $transaction->details = json_encode([
                'from_wallet' => $fromWallet == 1 ? 'main' : 'profit',
                'to_wallet' => $toWallet == 1 ? 'main' : 'profit'
            ]);
            $transaction->save();

            return response()->json([
                'success' => true,
                'message' => 'Wallet exchange completed successfully!'
            ]);

        } catch (\Exception $e) {
            Log::error('Wallet Exchange Error: ' . $e->getMessage(), [
                'line' => $e->getLine(),
                'file' => $e->getFile(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Exchange Error: ' . $e->getMessage()
            ]);
        }
    }
}




