<?php

/**
 * Investment Timer Fix Script
 * Run this script to fix all investment timers
 */

require_once 'vendor/autoload.php';

use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "=== Investment Timer Fix Script ===\n";

try {
    // Get site timezone
    $siteTimezone = setting('site_timezone', 'global') ?? 'UTC';
    echo "Site Timezone: {$siteTimezone}\n";
    
    // Get current time in site timezone
    $now = Carbon::now($siteTimezone);
    echo "Current Time: {$now}\n";
    
    // Get all ongoing investments
    $investments = DB::table('invests')
        ->join('schemas', 'invests.schema_id', '=', 'schemas.id')
        ->leftJoin('schedules', 'schemas.schedule_id', '=', 'schedules.id')
        ->where('invests.status', 'ongoing')
        ->select('invests.*', 'schedules.time as schedule_time', 'schemas.name as schema_name')
        ->get();
    
    echo "Found {$investments->count()} ongoing investments\n\n";
    
    $fixedCount = 0;
    
    foreach ($investments as $investment) {
        echo "Processing Investment ID: {$investment->id}\n";
        echo "  - Schema: {$investment->schema_name}\n";
        echo "  - Current next_profit_time: {$investment->next_profit_time}\n";
        
        // Get schedule time (in minutes)
        $scheduleMinutes = $investment->schedule_time ?? 1440; // Default 24 hours
        $periodHours = round($scheduleMinutes / 60, 1);
        
        // Calculate correct next profit time
        $correctNextProfitTime = $now->copy()->addMinutes($scheduleMinutes);
        
        // Calculate correct last profit time
        $correctLastProfitTime = $investment->last_profit_time ?? $investment->created_at;
        
        echo "  - Schedule Minutes: {$scheduleMinutes}\n";
        echo "  - Period Hours: {$periodHours}\n";
        echo "  - New next_profit_time: {$correctNextProfitTime}\n";
        echo "  - New last_profit_time: {$correctLastProfitTime}\n";
        
        // Update the investment
        DB::table('invests')
            ->where('id', $investment->id)
            ->update([
                'next_profit_time' => $correctNextProfitTime,
                'last_profit_time' => $correctLastProfitTime,
                'period_hours' => $periodHours,
                'updated_at' => now(),
            ]);
        
        echo "  ✅ Updated successfully\n\n";
        $fixedCount++;
    }
    
    echo "=== Summary ===\n";
    echo "Fixed: {$fixedCount} investments\n";
    echo "Script completed successfully!\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}
