@extends('frontend::layouts.user')
@section('title')
    {{ __('Investment Portfolio') }}
@endsection

@section('content')
<div class="invest-logs-container">
    <!-- Hero Section -->
    <div class="invest-hero">
        <div class="hero-content">
            <div class="hero-icon">📈</div>
            <h1 class="hero-title">{{ __('Investment Portfolio') }}</h1>
            <p class="hero-subtitle">{{ __('Track your investments, monitor performance, and manage your portfolio') }}</p>
        </div>
    </div>

    <!-- Main Content -->
    <div class="invest-main">
        <!-- Search and Filter Section -->
        <div class="glass-card filter-card">
            <div class="filter-content">
                <form action="" class="search-form">
                    <div class="search-input-group">
                        <div class="search-input">
                            <input type="text" name="query" value="{{ request('query') }}" placeholder="Search investments...">
                            <div class="search-icon">
                                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M21 21L16.514 16.506L21 21ZM19 10.5C19 15.194 15.194 19 10.5 19C5.806 19 2 15.194 2 10.5C2 5.806 5.806 2 10.5 2C15.194 2 19 5.806 19 10.5Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                </svg>
                            </div>
                        </div>
                        <button type="submit" class="search-btn">Search</button>
                    </div>
                </form>
                
                <div class="filter-controls">
                    <div class="limit-selector">
                        <label>Show:</label>
                        <select name="limit" onchange="this.form.submit()">
                            <option value="10" {{ request('limit') == 10 ? 'selected' : '' }}>10</option>
                            <option value="15" {{ request('limit') == 15 ? 'selected' : '' }}>15</option>
                            <option value="20" {{ request('limit') == 20 ? 'selected' : '' }}>20</option>
                            <option value="25" {{ request('limit') == 25 ? 'selected' : '' }}>25</option>
                            <option value="30" {{ request('limit') == 30 ? 'selected' : '' }}>30</option>
                        </select>
                    </div>
                </div>
            </div>
        </div>

        <!-- Investment Cards -->
        @php
            $logs = $data->when(request('query'),function($query){
                $query->whereHas('schema',function($schemaQuery){
                    $schemaQuery->where('name','LIKE','%'.request('query').'%');
                });
            })->paginate(request()->integer('limit',15))->withQueryString();
        @endphp

        @if(count($logs) > 0)
            <div class="investments-grid">
                @foreach ($logs as $invest)
                <div class="investment-card">
                    <div class="card-header">
                        <div class="investment-icon">
                            <div class="icon-bg">
                                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12 2L2 7L12 12L22 7L12 2Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                    <path d="M2 17L12 22L22 17" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                    <path d="M2 12L12 17L22 12" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                </svg>
                            </div>
                        </div>
                        <div class="investment-info">
                            <h3 class="investment-name">{{ $invest->schema->name }}</h3>
                            <p class="investment-date">{{ \Carbon\Carbon::parse($invest->created_at)->format('M d, Y') }}</p>
                        </div>
                        <div class="investment-amount">
                            <span class="amount">{{ $currencySymbol }}{{ number_format($invest->invest_amount, 2) }}</span>
                        </div>
                    </div>

                    <div class="card-body">
                        <div class="investment-stats">
                            <div class="stat-item">
                                <span class="stat-label">{{ __('ROI') }}</span>
                                <span class="stat-value roi">
                                    {{ $invest->interest_type == 'percentage' ? $invest->interest.'%' : $currencySymbol.$invest->interest }}
                                </span>
                            </div>
                            
                            @php
                                $calculateInterest = ($invest->interest*$invest->invest_amount)/100;
                                $interest = $invest->interest_type != 'percentage' ? $invest->interest : $calculateInterest;
                            @endphp
                            
                            <div class="stat-item">
                                <span class="stat-label">{{ __('Profit') }}</span>
                                <span class="stat-value profit">
                                    {{ $invest->already_return_profit .' x '.$interest .' = '. ($invest->already_return_profit*$interest).' '. $currency }}
                                </span>
                            </div>
                            
                            <div class="stat-item">
                                <span class="stat-label">{{ __('Periods') }}</span>
                                <span class="stat-value">
                                    {{ $invest->return_type != 'period' ? __('Unlimited') : $invest->number_of_period . ($invest->number_of_period < 2 ? ' Time' : ' Times') }}
                                </span>
                            </div>
                            
                            <div class="stat-item">
                                <span class="stat-label">{{ __('Capital Back') }}</span>
                                <span class="stat-value {{ $invest->capital_back ? 'success' : 'warning' }}">
                                    {{ $invest->capital_back ? 'Yes' : 'No' }}
                                </span>
                            </div>
                        </div>

                        @if($invest->status->value == 'ongoing')
                            <div class="investment-timeline">
                                <div class="timeline-header">
                                    <h4>{{ __('Next Profit') }}</h4>
                                    <div class="countdown" data-invest-id="{{ $invest->id }}" data-start="{{ $invest->last_profit_time ?? $invest->created_at }}" data-end="{{ $invest->next_profit_time }}">
                                        <span class="days" data-invest-id="{{ $invest->id }}" data-start="{{ $invest->last_profit_time ?? $invest->created_at }}" data-end="{{ $invest->next_profit_time }}">0</span>d 
                                        <span class="hours" data-invest-id="{{ $invest->id }}" data-start="{{ $invest->last_profit_time ?? $invest->created_at }}" data-end="{{ $invest->next_profit_time }}">0</span>h 
                                        <span class="minutes" data-invest-id="{{ $invest->id }}" data-start="{{ $invest->last_profit_time ?? $invest->created_at }}" data-end="{{ $invest->next_profit_time }}">0</span>m 
                                        <span class="seconds" data-invest-id="{{ $invest->id }}" data-start="{{ $invest->last_profit_time ?? $invest->created_at }}" data-end="{{ $invest->next_profit_time }}">0</span>s
                                    </div>
                                </div>
                                
                                <div class="progress-container">
                                    <div class="progress-bar">
                                        <div class="progress-fill" id="time-progress-{{ $invest->id }}" style="width: 0%"></div>
                                    </div>
                                    <span class="progress-text" id="percent-text-{{ $invest->id }}">0%</span>
                                </div>
                            </div>
                        @endif
                    </div>

                    <div class="card-footer">
                        <div class="status-badge {{ $invest->status->value }}">
                            @if($invest->status->value == 'ongoing')
                                <span class="badge ongoing">{{ __('Active') }}</span>
                            @elseif($invest->status->value == 'pending')
                                <span class="badge pending">{{ __('Pending') }}</span>
                            @elseif($invest->status->value == 'completed')
                                <span class="badge completed">{{ __('Completed') }}</span>
                            @else
                                <span class="badge cancelled">{{ __('Cancelled') }}</span>
                            @endif
                        </div>
                        
                        @if($invest->status->value == 'ongoing')
                            <button class="cancel-btn" onclick="confirmCancel({{ $invest->id }})">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M18 6L6 18M6 6l12 12" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                </svg>
                                {{ __('Cancel') }}
                            </button>
                        @endif
                    </div>
                </div>
                @endforeach
            </div>

            <!-- Pagination -->
            @if($logs->hasPages())
                <div class="pagination-wrapper">
                    {{ $logs->links() }}
                </div>
            @endif
        @else
            <div class="empty-state">
                <div class="empty-icon">📊</div>
                <h4 class="empty-title">{{ __('No Investments Found') }}</h4>
                <p class="empty-desc">{{ __('Start investing to see your portfolio here') }}</p>
                <div class="empty-action">
                    <a href="{{ route('user.schema.index') }}" class="action-btn">{{ __('Start Investing') }}</a>
                </div>
            </div>
        @endif
    </div>
</div>

<style>
/* Investment Logs Styles */
.invest-logs-container {
    min-height: 100vh;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 20px;
}

.invest-hero {
    text-align: center;
    margin-bottom: 40px;
    padding: 40px 20px;
}

.hero-content {
    max-width: 600px;
    margin: 0 auto;
}

.hero-icon {
    font-size: 4rem;
    margin-bottom: 20px;
    animation: float 3s ease-in-out infinite;
}

.hero-title {
    font-size: 3rem;
    font-weight: 700;
    color: white;
    margin-bottom: 15px;
    text-shadow: 0 4px 8px rgba(0,0,0,0.3);
}

.hero-subtitle {
    font-size: 1.2rem;
    color: rgba(255,255,255,0.9);
    line-height: 1.6;
}

.invest-main {
    max-width: 1200px;
    margin: 0 auto;
    display: flex;
    flex-direction: column;
    gap: 30px;
}

.glass-card {
    background: rgba(255,255,255,0.1);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    border: 1px solid rgba(255,255,255,0.2);
    padding: 30px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
}

/* Filter Card */
.filter-card {
    margin-bottom: 30px;
}

.filter-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 20px;
}

.search-form {
    flex: 1;
}

.search-input-group {
    display: flex;
    gap: 15px;
    align-items: center;
}

.search-input {
    position: relative;
    flex: 1;
}

.search-input input {
    width: 100%;
    background: rgba(255,255,255,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    border-radius: 15px;
    padding: 15px 50px 15px 20px;
    color: white;
    font-size: 1rem;
    transition: all 0.3s ease;
}

.search-input input::placeholder {
    color: rgba(255,255,255,0.7);
}

.search-input input:focus {
    outline: none;
    border-color: rgba(255,255,255,0.5);
    box-shadow: 0 0 20px rgba(255,255,255,0.1);
}

.search-icon {
    position: absolute;
    right: 15px;
    top: 50%;
    transform: translateY(-50%);
    color: rgba(255,255,255,0.7);
}

.search-btn {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border: none;
    padding: 15px 25px;
    border-radius: 15px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.search-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.filter-controls {
    display: flex;
    align-items: center;
    gap: 15px;
}

.limit-selector {
    display: flex;
    align-items: center;
    gap: 10px;
    color: white;
}

.limit-selector label {
    font-size: 0.9rem;
    font-weight: 500;
}

.limit-selector select {
    background: rgba(255,255,255,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    border-radius: 10px;
    padding: 8px 15px;
    color: white;
    font-size: 0.9rem;
}

/* Investment Cards */
.investments-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 25px;
}

.investment-card {
    background: rgba(255,255,255,0.08);
    border: 1px solid rgba(255,255,255,0.15);
    border-radius: 20px;
    padding: 25px;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.investment-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 20px 40px rgba(0,0,0,0.2);
    border-color: rgba(255,255,255,0.3);
}

.investment-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.1), transparent);
    transition: left 0.5s;
}

.investment-card:hover::before {
    left: 100%;
}

.card-header {
    display: flex;
    align-items: center;
    gap: 15px;
    margin-bottom: 20px;
}

.investment-icon {
    flex-shrink: 0;
}

.icon-bg {
    width: 50px;
    height: 50px;
    background: linear-gradient(135deg, #4ecdc4, #44a08d);
    border-radius: 15px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
}

.investment-info {
    flex: 1;
}

.investment-name {
    color: white;
    font-size: 1.2rem;
    font-weight: 600;
    margin-bottom: 5px;
}

.investment-date {
    color: rgba(255,255,255,0.7);
    font-size: 0.9rem;
}

.investment-amount {
    text-align: right;
}

.amount {
    color: #4caf50;
    font-size: 1.3rem;
    font-weight: 700;
}

.card-body {
    margin-bottom: 20px;
}

.investment-stats {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 15px;
    margin-bottom: 20px;
}

.stat-item {
    display: flex;
    flex-direction: column;
    gap: 5px;
}

.stat-label {
    color: rgba(255,255,255,0.7);
    font-size: 0.85rem;
    font-weight: 500;
}

.stat-value {
    color: white;
    font-size: 0.95rem;
    font-weight: 600;
}

.stat-value.roi {
    color: #4caf50;
}

.stat-value.profit {
    color: #ff9800;
}

.stat-value.success {
    color: #4caf50;
}

.stat-value.warning {
    color: #ff9800;
}

/* Timeline Section */
.investment-timeline {
    background: rgba(255,255,255,0.05);
    border-radius: 15px;
    padding: 20px;
    margin-top: 20px;
}

.timeline-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
}

.timeline-header h4 {
    color: white;
    font-size: 1rem;
    font-weight: 600;
    margin: 0;
}

.countdown {
    color: #4caf50;
    font-weight: 600;
    font-size: 0.9rem;
}

.progress-container {
    display: flex;
    align-items: center;
    gap: 15px;
}

.progress-bar {
    flex: 1;
    height: 8px;
    background: rgba(255,255,255,0.1);
    border-radius: 10px;
    overflow: hidden;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #4caf50, #8bc34a);
    border-radius: 10px;
    transition: width 0.5s ease;
    position: relative;
}

.progress-fill::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    animation: shimmer 2s infinite;
}

.progress-text {
    color: rgba(255,255,255,0.8);
    font-size: 0.85rem;
    font-weight: 600;
    min-width: 40px;
}

/* Card Footer */
.card-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: 20px;
    border-top: 1px solid rgba(255,255,255,0.1);
}

.status-badge {
    display: flex;
    align-items: center;
}

.badge {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.badge.ongoing {
    background: rgba(76, 175, 80, 0.2);
    color: #4caf50;
    border: 1px solid rgba(76, 175, 80, 0.3);
}

.badge.pending {
    background: rgba(255, 152, 0, 0.2);
    color: #ff9800;
    border: 1px solid rgba(255, 152, 0, 0.3);
}

.badge.completed {
    background: rgba(76, 175, 80, 0.2);
    color: #4caf50;
    border: 1px solid rgba(76, 175, 80, 0.3);
}

.badge.cancelled {
    background: rgba(244, 67, 54, 0.2);
    color: #f44336;
    border: 1px solid rgba(244, 67, 54, 0.3);
}

.cancel-btn {
    background: linear-gradient(135deg, #f44336, #d32f2f);
    color: white;
    border: none;
    padding: 8px 15px;
    border-radius: 10px;
    font-size: 0.8rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 5px;
}

.cancel-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 20px rgba(244, 67, 54, 0.3);
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 80px 20px;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 20px;
    opacity: 0.7;
}

.empty-title {
    color: white;
    font-size: 1.8rem;
    font-weight: 600;
    margin-bottom: 15px;
}

.empty-desc {
    color: rgba(255,255,255,0.8);
    font-size: 1.1rem;
    margin-bottom: 30px;
    line-height: 1.6;
}

.empty-action {
    display: flex;
    justify-content: center;
}

.action-btn {
    display: inline-block;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    text-decoration: none;
    padding: 15px 30px;
    border-radius: 15px;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.action-btn:hover {
    transform: translateY(-3px);
    box-shadow: 0 12px 35px rgba(102, 126, 234, 0.4);
    color: white;
}

/* Pagination */
.pagination-wrapper {
    display: flex;
    justify-content: center;
    margin-top: 30px;
}

.pagination-wrapper .pagination {
    display: flex;
    gap: 10px;
}

.pagination-wrapper .page-link {
    background: rgba(255,255,255,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    color: white;
    padding: 10px 15px;
    border-radius: 10px;
    text-decoration: none;
    transition: all 0.3s ease;
}

.pagination-wrapper .page-link:hover {
    background: rgba(255,255,255,0.2);
    transform: translateY(-2px);
}

.pagination-wrapper .page-item.active .page-link {
    background: #667eea;
    border-color: #667eea;
}

.pagination-wrapper .page-item.disabled .page-link {
    opacity: 0.5;
    cursor: not-allowed;
}

/* Animations */
@keyframes float {
    0%, 100% { transform: translateY(0px); }
    50% { transform: translateY(-10px); }
}

@keyframes shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

/* Responsive Design */
@media (max-width: 768px) {
    .invest-logs-container {
        padding: 15px;
    }
    
    .hero-title {
        font-size: 2rem;
    }
    
    .hero-subtitle {
        font-size: 1rem;
    }
    
    .glass-card {
        padding: 20px;
    }
    
    .filter-content {
        flex-direction: column;
        gap: 15px;
    }
    
    .search-input-group {
        flex-direction: column;
    }
    
    .investments-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }
    
    .investment-stats {
        grid-template-columns: 1fr;
        gap: 10px;
    }
    
    .card-header {
        flex-direction: column;
        text-align: center;
        gap: 10px;
    }
    
    .investment-amount {
        text-align: center;
    }
}

@media (max-width: 480px) {
    .hero-icon {
        font-size: 3rem;
    }
    
    .hero-title {
        font-size: 1.8rem;
    }
    
    .investment-card {
        padding: 20px;
    }
    
    .timeline-header {
        flex-direction: column;
        gap: 10px;
        text-align: center;
    }
    
    .progress-container {
        flex-direction: column;
        gap: 10px;
    }
    
    .card-footer {
        flex-direction: column;
        gap: 15px;
    }
    
    .empty-icon {
        font-size: 3rem;
    }
    
    .empty-title {
        font-size: 1.5rem;
    }
    
    .action-btn {
        padding: 12px 25px;
        font-size: 0.9rem;
    }
}
</style>

@endsection

@section('script')
<script>
// Set the timezone
const timezone = @json(setting('site_timezone','global'));

function updateCountdowns() {
    document.querySelectorAll('.days, .hours, .minutes, .seconds').forEach(element => {
        const investId = element.getAttribute('data-invest-id');
        const startTimeStr = element.getAttribute('data-start');
        const endTimeStr = element.getAttribute('data-end');

        const startTime = new Date(startTimeStr).getTime();
        const endTime = new Date(endTimeStr).getTime();

        const nowUtcStr = new Date().toLocaleString("en-US", {timeZone: timezone});
        const now = new Date(nowUtcStr).getTime();

        const distance = endTime - now >= 0 ? endTime - now : 0;
        const progressPercent = Math.min(100, ((now - startTime) / (endTime - startTime)) * 100).toFixed(2);

        // Update progress bar
        const progressBar = document.getElementById(`time-progress-${investId}`);
        if (progressBar) {
            progressBar.style.width = progressPercent + '%';
        }

        // Update percentage text
        const percentText = document.getElementById(`percent-text-${investId}`);
        if (percentText) {
            percentText.innerText = progressPercent + '%';
        }

        // Update countdown
        document.querySelector(`[data-invest-id="${investId}"].days`).innerText = Math.floor(distance / (1000 * 60 * 60 * 24));
        document.querySelector(`[data-invest-id="${investId}"].hours`).innerText = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        document.querySelector(`[data-invest-id="${investId}"].minutes`).innerText = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
        document.querySelector(`[data-invest-id="${investId}"].seconds`).innerText = Math.floor((distance % (1000 * 60)) / 1000);
    });
}

function confirmCancel(investId) {
    if (confirm('{{ __("Are you sure you want to cancel this investment?") }}')) {
        window.location.href = `/user/invest-cancel/${investId}`;
    }
}

// ROI Collection function
function collectROI(investId) {
    if (confirm('Are you sure you want to collect ROI for this investment?')) {
        // Show loading state
        const button = document.querySelector(`[data-id="${investId}"]`);
        const originalContent = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Collecting...';
        button.disabled = true;
        
        // Make AJAX request
        fetch('{{ route("user.invest.collect-roi") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({
                invest_id: investId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Show success message
                alert(data.message);
                
                // Update button to collected state
                button.innerHTML = '<i class="fas fa-check"></i> ROI Collected';
                button.className = 'btn btn-sm btn-success';
                button.disabled = true;
                
                // Reload page to update data
                setTimeout(() => {
                    location.reload();
                }, 1000);
            } else {
                // Show error message
                alert(data.message);
                
                // Restore button
                button.innerHTML = originalContent;
                button.disabled = false;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred. Please try again.');
            
            // Restore button
            button.innerHTML = originalContent;
            button.disabled = false;
        });
    }
}

// Add event listeners for ROI collection buttons
document.addEventListener('DOMContentLoaded', function() {
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('collect-roi-btn')) {
            const investId = e.target.getAttribute('data-id');
            collectROI(investId);
        }
    });
});

// Update every second
setInterval(updateCountdowns, 1000);

// Initial call
updateCountdowns();
</script>
@endsection
