@extends('frontend.shahdeveloper.layouts.app')

@section('content')
<div class="transfer-page">
    <!-- Header Section -->
    <div class="transfer-header">
        <div class="header-content">
            <h2>Money Transfer</h2>
            <p>Send money to other users securely</p>
        </div>
        <div class="wallet-balances">
            <div class="wallet-balance">
                <div class="wallet-icon main-wallet">
                    <i class="fas fa-wallet"></i>
                </div>
                <div class="wallet-info">
                    <span class="wallet-name">Main Wallet</span>
                    <span class="wallet-amount">{{ setting('site_currency_symbol', 'global', '$') }}{{ number_format(auth()->user()->balance, 2) }}</span>
                </div>
            </div>
            <div class="wallet-balance">
                <div class="wallet-icon profit-wallet">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="wallet-info">
                    <span class="wallet-name">Profit Wallet</span>
                    <span class="wallet-amount">{{ setting('site_currency_symbol', 'global', '$') }}{{ number_format(auth()->user()->profit_balance, 2) }}</span>
                </div>
            </div>
        </div>
    </div>

    <!-- Transfer Form -->
    <div class="transfer-form-container">
        <form id="transferForm" class="transfer-form">
            @csrf
            
            <!-- Recipient Selection -->
            <div class="form-group">
                <label for="recipient">Recipient Username/Email</label>
                <div class="input-with-search">
                    <input type="text" id="recipient" name="recipient" class="form-control" placeholder="Enter username or email" required>
                    <button type="button" class="search-btn" onclick="searchUser()">
                        <i class="fas fa-search"></i>
                    </button>
                </div>
                <div id="userSearchResult" class="user-search-result"></div>
            </div>

            <!-- Wallet Selection -->
            <div class="form-group">
                <label>Select Wallet</label>
                <div class="wallet-selection">
                    <div class="wallet-option" data-wallet="main">
                        <div class="wallet-option-content">
                            <div class="wallet-icon main-wallet">
                                <i class="fas fa-wallet"></i>
                            </div>
                            <div class="wallet-details">
                                <span class="wallet-name">Main Wallet</span>
                                <span class="wallet-balance">{{ setting('site_currency_symbol', 'global', '$') }}{{ number_format(auth()->user()->balance, 2) }}</span>
                            </div>
                        </div>
                        <div class="wallet-radio">
                            <input type="radio" name="wallet_type" value="main" id="wallet_main" checked>
                            <label for="wallet_main"></label>
                        </div>
                    </div>
                    
                    <div class="wallet-option" data-wallet="profit">
                        <div class="wallet-option-content">
                            <div class="wallet-icon profit-wallet">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <div class="wallet-details">
                                <span class="wallet-name">Profit Wallet</span>
                                <span class="wallet-balance">{{ setting('site_currency_symbol', 'global', '$') }}{{ number_format(auth()->user()->profit_balance, 2) }}</span>
                            </div>
                        </div>
                        <div class="wallet-radio">
                            <input type="radio" name="wallet_type" value="profit" id="wallet_profit">
                            <label for="wallet_profit"></label>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Amount Input -->
            <div class="form-group">
                <label for="amount">Amount</label>
                <div class="amount-input-group">
                    <span class="currency-symbol">{{ setting('site_currency_symbol', 'global', '$') }}</span>
                    <input type="number" id="amount" name="amount" class="form-control" placeholder="0.00" step="0.01" min="0.01" required>
                </div>
                <div class="amount-suggestions">
                    <button type="button" class="amount-btn" onclick="setAmount(10)">$10</button>
                    <button type="button" class="amount-btn" onclick="setAmount(50)">$50</button>
                    <button type="button" class="amount-btn" onclick="setAmount(100)">$100</button>
                    <button type="button" class="amount-btn" onclick="setAmount(500)">$500</button>
                </div>
            </div>

            <!-- Note -->
            <div class="form-group">
                <label for="note">Note (Optional)</label>
                <textarea id="note" name="note" class="form-control" placeholder="Add a note for this transfer" rows="3"></textarea>
            </div>

            <!-- Account Password -->
            <div class="form-group">
                <label for="password">Account Password</label>
                <div class="pin-input-group">
                    <input type="password" id="password" name="password" class="form-control" placeholder="Enter your account password" required>
                    <button type="button" class="pin-toggle" onclick="togglePin()">
                        <i class="fas fa-eye"></i>
                    </button>
                </div>
            </div>

            <!-- Transfer Button -->
            <button type="submit" class="btn-transfer">
                <i class="fas fa-paper-plane"></i>
                Send Transfer
            </button>
            <div id="transferStatus" class="inline-status" style="display:none;margin-top:10px;"></div>
        </form>
    </div>

    <!-- Quick Actions -->
    <div class="quick-actions">
        <a href="{{ route('user.shah.transfer.history') }}" class="quick-action-btn">
            <i class="fas fa-history"></i>
            <span>Transfer History</span>
        </a>
        <a href="#" class="quick-action-btn" onclick="showWalletExchange()">
            <i class="fas fa-exchange-alt"></i>
            <span>Wallet Exchange</span>
        </a>
    </div>

    <!-- Transfer Info -->
    <div class="transfer-info">
        <h3>Transfer Information</h3>
        <div class="info-cards">
            <div class="info-card">
                <i class="fas fa-shield-alt"></i>
                <h4>Secure</h4>
                <p>All transfers are encrypted and secure</p>
            </div>
            <div class="info-card">
                <i class="fas fa-clock"></i>
                <h4>Instant</h4>
                <p>Transfers are processed instantly</p>
            </div>
            <div class="info-card">
                <i class="fas fa-dollar-sign"></i>
                <h4>No Fees</h4>
                <p>Zero transfer fees for all users</p>
            </div>
        </div>
    </div>
</div>

<!-- Wallet Exchange Modal -->
<div id="walletExchangeModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Wallet Exchange</h3>
            <button class="modal-close" onclick="closeWalletExchange()">&times;</button>
        </div>
        <div class="modal-body">
            <form id="walletExchangeForm">
                @csrf
                <div class="form-group">
                    <label>From Wallet</label>
                    <select name="from_wallet" class="form-control" required>
                        <option value="1">Main Wallet</option>
                        <option value="2">Profit Wallet</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>To Wallet</label>
                    <select name="to_wallet" class="form-control" required>
                        <option value="2">Profit Wallet</option>
                        <option value="1">Main Wallet</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Amount</label>
                    <input type="number" name="amount" class="form-control" placeholder="0.00" step="0.01" min="0.01" required>
                </div>
                <button type="submit" class="btn-exchange">
                    <i class="fas fa-exchange-alt"></i>
                    Exchange
                </button>
            </form>
        </div>
    </div>
</div>
@endsection

@section('style')
<style>
/* Mobile App-like Professional Design */
.transfer-page {
    padding: 0;
    max-width: 100%;
    margin: 0;
    background: linear-gradient(135deg, #0a0a0a 0%, #1a1a1a 100%);
    min-height: 100vh;
}

/* Header Section */
.transfer-header {
    background: linear-gradient(135deg, rgba(0, 255, 136, 0.1) 0%, rgba(0, 224, 163, 0.05) 100%);
    padding: 30px 20px;
    border-bottom: 1px solid rgba(0, 255, 136, 0.2);
}

.header-content {
    text-align: center;
    margin-bottom: 25px;
}

.header-content h2 {
    color: #ffffff;
    font-size: 2.2rem;
    font-weight: 700;
    margin-bottom: 8px;
    text-shadow: 0 2px 10px rgba(0, 255, 136, 0.3);
}

.header-content p {
    color: rgba(255, 255, 255, 0.8);
    font-size: 1rem;
    font-weight: 400;
}

.wallet-balances {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 15px;
    max-width: 400px;
    margin: 0 auto;
}

.wallet-balance {
    background: rgba(255, 255, 255, 0.08);
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    border-radius: 16px;
    border: 1px solid rgba(255, 255, 255, 0.1);
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    transition: all 0.3s ease;
}

.wallet-balance:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 255, 136, 0.15);
}

.wallet-icon {
    width: 50px;
    height: 50px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
}

.wallet-icon.main-wallet {
    background: linear-gradient(135deg, #00ff88, #00e0a3);
    color: #001f11;
}

.wallet-icon.profit-wallet {
    background: linear-gradient(135deg, #ff6b6b, #ff8e8e);
    color: #ffffff;
}

.wallet-info {
    flex: 1;
}

.wallet-name {
    display: block;
    color: rgba(255, 255, 255, 0.9);
    font-size: 0.9rem;
    font-weight: 500;
    margin-bottom: 4px;
}

.wallet-amount {
    display: block;
    color: #ffffff;
    font-size: 1.2rem;
    font-weight: 700;
}

/* Form Container */
.transfer-form-container {
    background: rgba(255, 255, 255, 0.03);
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    border-radius: 24px;
    border: 1px solid rgba(255, 255, 255, 0.1);
    margin: 20px;
    padding: 30px;
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
}

.transfer-form {
    display: flex;
    flex-direction: column;
    gap: 25px;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.form-group label {
    color: #ffffff;
    font-weight: 600;
    font-size: 1rem;
    margin-bottom: 8px;
}

/* Input Styles */
.form-control {
    background: rgba(255, 255, 255, 0.08);
    border: 2px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    padding: 16px 20px;
    color: #ffffff;
    font-size: 1rem;
    font-weight: 500;
    transition: all 0.3s ease;
    width: 100%;
}

.form-control:focus {
    outline: none;
    border-color: #00ff88;
    box-shadow: 0 0 0 4px rgba(0, 255, 136, 0.1);
    background: rgba(255, 255, 255, 0.12);
}

.form-control::placeholder {
    color: rgba(255, 255, 255, 0.5);
}

/* Search Input */
.input-with-search {
    position: relative;
    display: flex;
    align-items: center;
}

.search-btn {
    position: absolute;
    right: 12px;
    background: rgba(0, 255, 136, 0.2);
    border: none;
    color: #00ff88;
    padding: 8px 12px;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.search-btn:hover {
    background: rgba(0, 255, 136, 0.3);
    transform: scale(1.05);
}

.user-search-result {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: rgba(0, 0, 0, 0.9);
    backdrop-filter: blur(20px);
    border-radius: 12px;
    border: 1px solid rgba(255, 255, 255, 0.1);
    margin-top: 5px;
    z-index: 1000;
    display: none;
}

/* Wallet Selection */
.wallet-selection {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.wallet-option {
    background: rgba(255, 255, 255, 0.05);
    border: 2px solid rgba(255, 255, 255, 0.1);
    border-radius: 16px;
    padding: 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}

.wallet-option:hover {
    border-color: rgba(0, 255, 136, 0.3);
    background: rgba(0, 255, 136, 0.05);
}

.wallet-option.selected {
    border-color: #00ff88;
    background: rgba(0, 255, 136, 0.1);
    box-shadow: 0 0 20px rgba(0, 255, 136, 0.2);
}

.wallet-option-content {
    display: flex;
    align-items: center;
    gap: 15px;
    flex: 1;
}

.wallet-details {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.wallet-radio {
    position: relative;
}

.wallet-radio input[type="radio"] {
    opacity: 0;
    position: absolute;
}

.wallet-radio label {
    width: 24px;
    height: 24px;
    border: 2px solid rgba(255, 255, 255, 0.3);
    border-radius: 50%;
    display: block;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}

.wallet-radio input[type="radio"]:checked + label {
    border-color: #00ff88;
    background: #00ff88;
}

.wallet-radio input[type="radio"]:checked + label::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 8px;
    height: 8px;
    background: #001f11;
    border-radius: 50%;
}

/* Amount Input */
.amount-input-group {
    position: relative;
    display: flex;
    align-items: center;
}

.currency-symbol {
    position: absolute;
    left: 20px;
    color: #00ff88;
    font-weight: 700;
    font-size: 1.2rem;
    z-index: 2;
}

.amount-input-group .form-control {
    padding-left: 50px;
    font-size: 1.2rem;
    font-weight: 700;
}

.amount-suggestions {
    display: flex;
    gap: 10px;
    margin-top: 10px;
    flex-wrap: wrap;
}

.amount-btn {
    background: rgba(0, 255, 136, 0.1);
    border: 1px solid rgba(0, 255, 136, 0.3);
    color: #00ff88;
    padding: 8px 16px;
    border-radius: 20px;
    font-size: 0.9rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.amount-btn:hover {
    background: rgba(0, 255, 136, 0.2);
    transform: translateY(-1px);
}

/* PIN Input */
.pin-input-group {
    position: relative;
    display: flex;
    align-items: center;
}

.pin-toggle {
    position: absolute;
    right: 16px;
    background: none;
    border: none;
    color: rgba(255, 255, 255, 0.6);
    cursor: pointer;
    padding: 8px;
    transition: all 0.3s ease;
}

.pin-toggle:hover {
    color: #00ff88;
}

/* Transfer Button */
.btn-transfer {
    background: linear-gradient(135deg, #00ff88, #00e0a3);
    border: none;
    color: #001f11;
    padding: 18px 30px;
    border-radius: 16px;
    font-weight: 700;
    font-size: 1.1rem;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 12px;
    margin-top: 20px;
    box-shadow: 0 8px 25px rgba(0, 255, 136, 0.3);
}

.btn-transfer:hover {
    transform: translateY(-3px);
    box-shadow: 0 12px 35px rgba(0, 255, 136, 0.4);
}

.btn-transfer:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

/* Quick Actions */
.quick-actions {
    display: flex;
    gap: 15px;
    margin: 20px;
    margin-bottom: 30px;
}

/* Inline status message */
.inline-status { padding: 10px 14px; border-radius: 10px; font-weight: 600; }
.inline-status.success { background: rgba(34,197,94,.15); border:1px solid rgba(34,197,94,.35); color:#b7ffcf; }
.inline-status.error { background: rgba(239,68,68,.12); border:1px solid rgba(239,68,68,.35); color:#ffc3c3; }

.quick-action-btn {
    flex: 1;
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(20px);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 16px;
    padding: 20px;
    text-decoration: none;
    color: #ffffff;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 10px;
    transition: all 0.3s ease;
}

.quick-action-btn:hover {
    background: rgba(0, 255, 136, 0.1);
    border-color: rgba(0, 255, 136, 0.3);
    transform: translateY(-2px);
    color: #ffffff;
    text-decoration: none;
}

.quick-action-btn i {
    font-size: 1.5rem;
    color: #00ff88;
}

.quick-action-btn span {
    font-weight: 600;
    font-size: 0.9rem;
}

/* Transfer Info */
.transfer-info {
    background: rgba(255, 255, 255, 0.03);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    border: 1px solid rgba(255, 255, 255, 0.1);
    margin: 20px;
    padding: 30px;
}

.transfer-info h3 {
    color: #ffffff;
    font-size: 1.4rem;
    margin-bottom: 25px;
    text-align: center;
    font-weight: 600;
}

.info-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 20px;
}

.info-card {
    text-align: center;
    padding: 25px 15px;
    background: rgba(0, 255, 136, 0.05);
    border-radius: 16px;
    border: 1px solid rgba(0, 255, 136, 0.1);
    transition: all 0.3s ease;
}

.info-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(0, 255, 136, 0.15);
}

.info-card i {
    font-size: 2.5rem;
    color: #00ff88;
    margin-bottom: 15px;
}

.info-card h4 {
    color: #ffffff;
    font-size: 1rem;
    margin-bottom: 8px;
    font-weight: 600;
}

.info-card p {
    color: rgba(255, 255, 255, 0.7);
    font-size: 0.85rem;
    margin: 0;
    line-height: 1.4;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 10000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
    backdrop-filter: blur(10px);
}

.modal-content {
    background: rgba(20, 20, 20, 0.95);
    backdrop-filter: blur(20px);
    margin: 5% auto;
    padding: 0;
    border-radius: 20px;
    border: 1px solid rgba(255, 255, 255, 0.1);
    width: 90%;
    max-width: 500px;
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.5);
}

.modal-header {
    padding: 25px 30px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-header h3 {
    color: #ffffff;
    font-size: 1.3rem;
    font-weight: 600;
    margin: 0;
}

.modal-close {
    background: none;
    border: none;
    color: rgba(255, 255, 255, 0.6);
    font-size: 2rem;
    cursor: pointer;
    transition: all 0.3s ease;
}

.modal-close:hover {
    color: #ff6b6b;
}

.modal-body {
    padding: 30px;
}

.btn-exchange {
    background: linear-gradient(135deg, #00ff88, #00e0a3);
    border: none;
    color: #001f11;
    padding: 15px 25px;
    border-radius: 12px;
    font-weight: 700;
    font-size: 1rem;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    width: 100%;
    margin-top: 20px;
}

.btn-exchange:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 255, 136, 0.3);
}

/* Responsive Design */
@media (max-width: 768px) {
    .transfer-page {
        padding: 0;
    }
    
    .transfer-header {
        padding: 25px 15px;
    }
    
    .header-content h2 {
        font-size: 1.8rem;
    }
    
    .wallet-balances {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .transfer-form-container {
        margin: 15px;
        padding: 25px;
    }
    
    .quick-actions {
        margin: 15px;
        flex-direction: column;
    }
    
    .transfer-info {
        margin: 15px;
        padding: 25px;
    }
    
    .info-cards {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .modal-content {
        width: 95%;
        margin: 10% auto;
    }
    
    .modal-header, .modal-body {
        padding: 20px;
    }
}

@media (max-width: 480px) {
    .transfer-form-container {
        margin: 10px;
        padding: 20px;
    }
    
    .wallet-option {
        padding: 15px;
    }
    
    .amount-suggestions {
        justify-content: center;
    }
    
    .amount-btn {
        padding: 6px 12px;
        font-size: 0.8rem;
    }
}
</style>
@endsection

@section('script')
<script>
// Wallet Selection
document.querySelectorAll('.wallet-option').forEach(option => {
    option.addEventListener('click', function() {
        // Remove selected class from all options
        document.querySelectorAll('.wallet-option').forEach(opt => opt.classList.remove('selected'));
        
        // Add selected class to clicked option
        this.classList.add('selected');
        
        // Check the radio button
        const radio = this.querySelector('input[type="radio"]');
        radio.checked = true;
        
        // Update wallet balances display
        updateWalletDisplay();
    });
});

// Amount suggestion buttons
function setAmount(amount) {
    document.getElementById('amount').value = amount;
    updateWalletDisplay();
}

// Toggle Password visibility
function togglePin() {
    const passwordInput = document.getElementById('password');
    const toggleBtn = document.querySelector('.pin-toggle i');
    
    if (passwordInput.type === 'password') {
        passwordInput.type = 'text';
        toggleBtn.className = 'fas fa-eye-slash';
    } else {
        passwordInput.type = 'password';
        toggleBtn.className = 'fas fa-eye';
    }
}

// Search user function
function searchUser() {
    const recipient = document.getElementById('recipient').value;
    const searchResult = document.getElementById('userSearchResult');
    
    if (recipient.length < 3) {
        searchResult.style.display = 'none';
        return;
    }
    
    // Simulate user search (you can implement actual search here)
    fetch(`/api/search-user?q=${encodeURIComponent(recipient)}`, {
        method: 'GET',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(async response => {
        const text = await response.text();
        try { return JSON.parse(text); } catch (e) { return { users: [] }; }
    })
    .then(data => {
        if (data.users && data.users.length > 0) {
            searchResult.innerHTML = data.users.map(user => 
                `<div class="user-result-item" onclick="selectUser('${user.username}', '${user.email}')">
                    <div class="user-avatar">${user.avatar ? `<img src="${user.avatar}" alt="${user.username}">` : '<i class="fas fa-user"></i>'}</div>
                    <div class="user-info">
                        <div class="user-name">${user.username}</div>
                        <div class="user-email">${user.email}</div>
                    </div>
                </div>`
            ).join('');
            searchResult.style.display = 'block';
        } else {
            searchResult.innerHTML = '<div class="no-results">No users found</div>';
            searchResult.style.display = 'block';
        }
    })
    .catch(error => {
        console.error('Search error:', error);
        searchResult.style.display = 'none';
    });
}

// Select user from search results
function selectUser(username, email) {
    document.getElementById('recipient').value = username;
    document.getElementById('userSearchResult').style.display = 'none';
}

// Update wallet display based on selection
function updateWalletDisplay() {
    const selectedWallet = document.querySelector('input[name="wallet_type"]:checked').value;
    const amount = parseFloat(document.getElementById('amount').value) || 0;
    
    // Update wallet balances in the form
    const walletOptions = document.querySelectorAll('.wallet-option');
    walletOptions.forEach(option => {
        const walletType = option.dataset.wallet;
        const balanceElement = option.querySelector('.wallet-balance');
        
        if (walletType === selectedWallet) {
            // Show remaining balance after transfer
            const currentBalance = walletType === 'main' ? 
                {{ auth()->user()->balance }} : 
                {{ auth()->user()->profit_balance }};
            const remainingBalance = currentBalance - amount;
            balanceElement.textContent = `{{ setting('site_currency_symbol', 'global', '$') }}${remainingBalance.toFixed(2)}`;
        }
    });
}

// Wallet Exchange Modal
function showWalletExchange() {
    document.getElementById('walletExchangeModal').style.display = 'block';
}

function closeWalletExchange() {
    document.getElementById('walletExchangeModal').style.display = 'none';
}

// Transfer Form Submission
document.getElementById('transferForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    const button = this.querySelector('button[type="submit"]');
    const originalText = button.innerHTML;
    
    // Show loading state
    button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
    button.disabled = true;
    
    fetch('{{ route("user.shah.transfer.submit") }}', {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Accept': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(async response => {
        // If backend returned validation redirect/html, try to parse JSON safely
        const text = await response.text();
        try { return JSON.parse(text); } catch (e) { return { success: false, message: text || 'Unexpected response' }; }
    })
    .then(data => {
        const statusEl = document.getElementById('transferStatus');
        if (data.success) {
            // Toast + inline success bar
            showSuccessMessage(data.message || 'Transfer successful');
            statusEl.className = 'inline-status success';
            statusEl.textContent = data.message || 'Transfer completed successfully!';
            statusEl.style.display = 'block';
            this.reset();
            updateWalletDisplay();
        } else {
            showErrorMessage(data.message || 'Transfer failed');
            statusEl.className = 'inline-status error';
            statusEl.textContent = data.message || 'Transfer failed';
            statusEl.style.display = 'block';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showErrorMessage('An error occurred while processing transfer');
    })
    .finally(() => {
        button.innerHTML = originalText;
        button.disabled = false;
    });
});

// Wallet Exchange Form Submission
document.getElementById('walletExchangeForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    const button = this.querySelector('button[type="submit"]');
    const originalText = button.innerHTML;
    
    // Show loading state
    button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Exchanging...';
    button.disabled = true;
    

    fetch('{{ route("user.shah.wallet.exchange") }}', {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccessMessage(data.message);
            this.reset();
            closeWalletExchange();
            // Refresh page to update balances
            setTimeout(() => location.reload(), 2000);
        } else {
            showErrorMessage(data.message || 'Exchange failed');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showErrorMessage('An error occurred while processing exchange');
    })
    .finally(() => {
        button.innerHTML = originalText;
        button.disabled = false;
    });
});

// Success Message
function showSuccessMessage(message) {
    const successDiv = document.createElement('div');
    successDiv.className = 'success-message';
    successDiv.innerHTML = `
        <div class="success-content">
            <i class="fas fa-check-circle"></i>
            <span>${message}</span>
        </div>
    `;
    
    document.body.appendChild(successDiv);
    
    // Animate in
    setTimeout(() => successDiv.classList.add('show'), 100);
    
    // Remove after 3 seconds
    setTimeout(() => {
        successDiv.classList.remove('show');
        setTimeout(() => document.body.removeChild(successDiv), 300);
    }, 3000);
}

// Error Message
function showErrorMessage(message) {
    const errorDiv = document.createElement('div');
    errorDiv.className = 'error-message';
    errorDiv.innerHTML = `
        <div class="error-content">
            <i class="fas fa-exclamation-circle"></i>
            <span>${message}</span>
        </div>
    `;
    
    document.body.appendChild(errorDiv);
    
    // Animate in
    setTimeout(() => errorDiv.classList.add('show'), 100);
    
    // Remove after 4 seconds
    setTimeout(() => {
        errorDiv.classList.remove('show');
        setTimeout(() => document.body.removeChild(errorDiv), 300);
    }, 4000);
}

// Close modals when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('walletExchangeModal');
    if (event.target === modal) {
        closeWalletExchange();
    }
}

// Initialize wallet display
document.addEventListener('DOMContentLoaded', function() {
    updateWalletDisplay();
    
    // Add event listeners for amount input
    document.getElementById('amount').addEventListener('input', updateWalletDisplay);
    
    // Add event listeners for wallet selection
    document.querySelectorAll('input[name="wallet_type"]').forEach(radio => {
        radio.addEventListener('change', updateWalletDisplay);
    });
});
</script>

<!-- Success/Error Message Styles -->
<style>
.success-message, .error-message {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 10001;
    transform: translateX(400px);
    transition: transform 0.3s ease;
}

.success-message.show, .error-message.show {
    transform: translateX(0);
}

.success-content, .error-content {
    background: rgba(0, 0, 0, 0.9);
    backdrop-filter: blur(20px);
    border-radius: 12px;
    padding: 15px 20px;
    display: flex;
    align-items: center;
    gap: 12px;
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.success-content {
    border-left: 4px solid #00ff88;
}

.error-content {
    border-left: 4px solid #ff6b6b;
}

.success-content i {
    color: #00ff88;
    font-size: 1.2rem;
}

.error-content i {
    color: #ff6b6b;
    font-size: 1.2rem;
}

.success-content span, .error-content span {
    color: #ffffff;
    font-weight: 500;
}

.user-result-item {
    padding: 12px 16px;
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
    transition: all 0.3s ease;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.user-result-item:hover {
    background: rgba(0, 255, 136, 0.1);
}

.user-result-item:last-child {
    border-bottom: none;
}

.user-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: rgba(0, 255, 136, 0.2);
    display: flex;
    align-items: center;
    justify-content: center;
    color: #00ff88;
}

.user-avatar img {
    width: 100%;
    height: 100%;
    border-radius: 50%;
    object-fit: cover;
}

.user-name {
    color: #ffffff;
    font-weight: 600;
    font-size: 0.9rem;
}

.user-email {
    color: rgba(255, 255, 255, 0.7);
    font-size: 0.8rem;
}

.no-results {
    padding: 20px;
    text-align: center;
    color: rgba(255, 255, 255, 0.7);
    font-style: italic;
}

// Close modal function
function closeModal() {
    const modal = document.getElementById('walletExchangeModal');
    if (modal) {
        modal.style.display = 'none';
    }
}

// Close wallet exchange modal function
function closeWalletExchangeModal() {
    const modal = document.getElementById('walletExchangeModal');
    if (modal) {
        modal.style.display = 'none';
    }
}
</style>
@endsection
