@extends('frontend.shahdeveloper.layouts.user')

@section('title', 'Verify OTP')

@section('content')
<div class="container-fluid">
    <div class="row justify-content-center">
        <div class="col-xl-6 col-lg-8">
            <!-- OTP Verification Card -->
            <div class="otp-verification-card">
                <div class="otp-header">
                    <div class="otp-icon">
                        <i class="fas fa-shield-alt"></i>
                    </div>
                    <h2 class="otp-title">Verify OTP</h2>
                    <p class="otp-subtitle">Enter the 6-digit code sent to your email address</p>
                </div>

                <form action="{{ route('user.withdraw.verify.otp') }}" method="POST" id="otpForm">
                    @csrf
                    <div class="otp-input-section">
                        <div class="otp-inputs">
                            <input type="text" name="otp_1" class="otp-input" maxlength="1" required>
                            <input type="text" name="otp_2" class="otp-input" maxlength="1" required>
                            <input type="text" name="otp_3" class="otp-input" maxlength="1" required>
                            <input type="text" name="otp_4" class="otp-input" maxlength="1" required>
                            <input type="text" name="otp_5" class="otp-input" maxlength="1" required>
                            <input type="text" name="otp_6" class="otp-input" maxlength="1" required>
                        </div>
                        <input type="hidden" name="otp" id="fullOtp">
                    </div>

                    <div class="otp-timer">
                        <p>Code expires in: <span id="timer">15:00</span></p>
                    </div>

                    <div class="otp-actions">
                        <button type="submit" class="btn btn-primary" id="verifyBtn">
                            <i class="fas fa-check"></i>
                            Verify OTP
                        </button>
                        
                        <button type="button" class="btn btn-outline-secondary" id="resendBtn" disabled>
                            <i class="fas fa-redo"></i>
                            Resend OTP
                        </button>
                    </div>
                </form>

                <div class="otp-help">
                    <p><i class="fas fa-info-circle"></i> Didn't receive the code? Check your spam folder or resend after the timer expires.</p>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.otp-verification-card {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    border: 1px solid rgba(0, 255, 136, 0.2);
    padding: 3rem 2rem;
    text-align: center;
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
}

.otp-header {
    margin-bottom: 2rem;
}

.otp-icon {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #00ff88, #00d4aa);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 1.5rem;
    animation: pulse 2s infinite;
}

.otp-icon i {
    font-size: 2.5rem;
    color: #001f11;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); }
}

.otp-title {
    color: var(--text-primary);
    font-size: 2rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    background: linear-gradient(135deg, #00ff88, #00d4aa);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.otp-subtitle {
    color: var(--text-secondary);
    font-size: 1.1rem;
    margin: 0;
}

.otp-input-section {
    margin-bottom: 2rem;
}

.otp-inputs {
    display: flex;
    justify-content: center;
    gap: 1rem;
    margin-bottom: 1rem;
}

.otp-input {
    width: 60px;
    height: 60px;
    background: rgba(255, 255, 255, 0.1);
    border: 2px solid rgba(255, 255, 255, 0.2);
    border-radius: 12px;
    color: var(--text-primary);
    font-size: 1.5rem;
    font-weight: 600;
    text-align: center;
    transition: all 0.3s ease;
}

.otp-input:focus {
    background: rgba(255, 255, 255, 0.15);
    border-color: var(--primary-color);
    color: var(--text-primary);
    box-shadow: 0 0 0 0.2rem rgba(0, 255, 136, 0.25);
    outline: none;
}

.otp-input.filled {
    border-color: var(--primary-color);
    background: rgba(0, 255, 136, 0.1);
}

.otp-timer {
    margin-bottom: 2rem;
}

.otp-timer p {
    color: var(--text-secondary);
    margin: 0;
    font-size: 1rem;
}

.otp-timer span {
    color: var(--primary-color);
    font-weight: 600;
    font-size: 1.1rem;
}

.otp-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    margin-bottom: 2rem;
}

.btn {
    padding: 12px 24px;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: none;
    font-size: 1rem;
    min-width: 150px;
    justify-content: center;
}

.btn-primary {
    background: linear-gradient(135deg, #00ff88, #00d4aa);
    color: #001f11;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 10px 20px rgba(0, 255, 136, 0.3);
    color: #001f11;
}

.btn-primary:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.btn-outline-secondary {
    background: transparent;
    color: var(--text-primary);
    border: 2px solid rgba(255, 255, 255, 0.3);
}

.btn-outline-secondary:hover:not(:disabled) {
    background: rgba(255, 255, 255, 0.1);
    color: var(--text-primary);
    transform: translateY(-2px);
}

.btn-outline-secondary:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.otp-help {
    background: rgba(0, 255, 136, 0.1);
    border: 1px solid rgba(0, 255, 136, 0.2);
    border-radius: 10px;
    padding: 1rem;
}

.otp-help p {
    color: var(--text-secondary);
    margin: 0;
    font-size: 0.9rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.otp-help i {
    color: var(--primary-color);
}

/* Responsive */
@media (max-width: 768px) {
    .otp-verification-card {
        padding: 2rem 1.5rem;
    }
    
    .otp-inputs {
        gap: 0.75rem;
    }
    
    .otp-input {
        width: 50px;
        height: 50px;
        font-size: 1.2rem;
    }
    
    .otp-actions {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .btn {
        min-width: auto;
        width: 100%;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const otpInputs = document.querySelectorAll('.otp-input');
    const fullOtpInput = document.getElementById('fullOtp');
    const verifyBtn = document.getElementById('verifyBtn');
    const resendBtn = document.getElementById('resendBtn');
    const timerElement = document.getElementById('timer');
    
    let timeLeft = 15 * 60; // 15 minutes in seconds
    let timerInterval;
    
    // Start timer
    startTimer();
    
    // Handle OTP input
    otpInputs.forEach((input, index) => {
        input.addEventListener('input', function(e) {
            const value = e.target.value;
            
            // Only allow numbers
            if (!/^\d$/.test(value)) {
                e.target.value = '';
                return;
            }
            
            // Add filled class
            e.target.classList.add('filled');
            
            // Move to next input
            if (value && index < otpInputs.length - 1) {
                otpInputs[index + 1].focus();
            }
            
            // Update full OTP
            updateFullOtp();
        });
        
        input.addEventListener('keydown', function(e) {
            // Handle backspace
            if (e.key === 'Backspace' && !e.target.value && index > 0) {
                otpInputs[index - 1].focus();
                otpInputs[index - 1].classList.remove('filled');
            }
        });
        
        input.addEventListener('paste', function(e) {
            e.preventDefault();
            const pastedData = e.clipboardData.getData('text');
            const digits = pastedData.replace(/\D/g, '').slice(0, 6);
            
            digits.split('').forEach((digit, i) => {
                if (otpInputs[i]) {
                    otpInputs[i].value = digit;
                    otpInputs[i].classList.add('filled');
                }
            });
            
            updateFullOtp();
        });
    });
    
    function updateFullOtp() {
        const otp = Array.from(otpInputs).map(input => input.value).join('');
        fullOtpInput.value = otp;
        
        // Enable/disable verify button
        verifyBtn.disabled = otp.length !== 6;
    }
    
    function startTimer() {
        timerInterval = setInterval(() => {
            const minutes = Math.floor(timeLeft / 60);
            const seconds = timeLeft % 60;
            
            timerElement.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            
            if (timeLeft <= 0) {
                clearInterval(timerInterval);
                timerElement.textContent = 'Expired';
                resendBtn.disabled = false;
                resendBtn.innerHTML = '<i class="fas fa-redo"></i> Resend OTP';
            }
            
            timeLeft--;
        }, 1000);
    }
    
    // Handle resend OTP
    resendBtn.addEventListener('click', function() {
        if (this.disabled) return;
        
        // Reset form
        otpInputs.forEach(input => {
            input.value = '';
            input.classList.remove('filled');
        });
        fullOtpInput.value = '';
        verifyBtn.disabled = true;
        
        // Reset timer
        timeLeft = 15 * 60;
        clearInterval(timerInterval);
        startTimer();
        
        // Disable resend button
        this.disabled = true;
        this.innerHTML = '<i class="fas fa-redo"></i> Resend OTP';
        
        // Make AJAX request to resend OTP
        fetch('{{ route("user.withdraw.resend-otp") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('OTP sent successfully!');
            } else {
                alert('Failed to resend OTP. Please try again.');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred. Please try again.');
        });
    });
    
    // Handle form submission
    document.getElementById('otpForm').addEventListener('submit', function(e) {
        const otp = fullOtpInput.value;
        if (otp.length !== 6) {
            e.preventDefault();
            alert('Please enter a complete 6-digit OTP');
            return;
        }
    });
});
</script>
@endsection
